(*#line 178.10 "mlb.lex"*)functor MLBLexFun (structure Tokens : MLB_TOKENS)(*#line 1.1 "mlb.lex.sml"*)
=
   struct
    structure UserDeclarations =
      struct
(*#line 1.1 "mlb.lex"*)(* Copyright (C) 2009,2016,2017,2024 Matthew Fluet.
 * Copyright (C) 2004-2006 Henry Cejtin, Matthew Fluet, Suresh
 *    Jagannathan, and Stephen Weeks.
 *
 * MLton is released under a HPND-style license.
 * See the file MLton-LICENSE for details.
 *)

type svalue = Tokens.svalue
type pos = SourcePos.t
type lexresult = (svalue, pos) Tokens.token
type lexarg = {source: Source.t}
type arg = lexarg
type ('a,'b) token = ('a,'b) Tokens.token

fun lastPos (yypos, yytext) = yypos + size yytext - 1

fun tok (t, x, s, l) =
   let
      val left = Source.getPos (s, l)
      val right = Source.getPos (s, lastPos (l, x))
   in
      t (left, right)
   end

fun tok' (t, x, s, l) = tok (fn (l, r) => t (x, l, r), x, s, l)

fun error' (left, right, msg) =
   Control.errorStr (Region.make {left = left, right = right}, msg)
fun error (source, left, right, msg) =
   error' (Source.getPos (source, left), Source.getPos (source, right), msg)


(* Comments *)
local
   val commentErrors: string list ref = ref []
   val commentLeft = ref SourcePos.bogus
   val commentStack: (int -> unit) list ref = ref []
in
   fun addCommentError msg =
      List.push (commentErrors, msg)
   val inComment = fn () => not (List.isEmpty (!commentStack))
   fun startComment (source, yypos, th) =
      if inComment ()
         then List.push (commentStack, fn _ => th ())
         else (commentErrors := []
               ; commentLeft := Source.getPos (source, yypos)
               ; List.push (commentStack, fn yypos =>
                            (List.foreach (!commentErrors, fn msg =>
                                           error' (!commentLeft,
                                                   Source.getPos (source, yypos),
                                                   msg))
                             ; th ())))
   fun finishComment yypos =
      (List.pop commentStack) yypos
end


(* Line Directives *)
local
   val lineDirCol: int ref = ref ~1
   val lineDirFile: File.t option ref = ref NONE
   val lineDirLine: int ref = ref ~1
in
   fun startLineDir (source, yypos, th) =
      let
         val _ = lineDirCol := ~1
         val _ = lineDirFile := NONE
         val _ = lineDirLine := ~1
      in
         startComment (source, yypos, th)
      end
   fun addLineDirLineCol (line, col) =
      let
         val _ = lineDirLine := line
         val _ = lineDirCol := col
      in
         ()
      end
   fun addLineDirFile file =
      let
         val _ = lineDirFile := SOME file
      in
         ()
      end
   fun finishLineDir (source, yypos) =
      let
         val col = !lineDirCol
         val file = !lineDirFile
         val line = !lineDirLine
         val _ = lineDirCol := ~1
         val _ = lineDirFile := NONE
         val _ = lineDirLine := ~1
      in
         finishComment yypos
         ; Source.lineDirective (source, file,
                                 {lineNum = line,
                                  lineStart = yypos + 1 - col})
      end
end


(* Text Constants *)
local
   val chars: char list ref = ref []
   val inText = ref false
   val textLeft = ref SourcePos.bogus
   val textFinishFn: (string * SourcePos.t * SourcePos.t -> lexresult) ref = ref (fn _ => raise Fail "textFinish")
in
   fun startText (tl, tf) =
      let
         val _ = chars := []
         val _ = inText := true
         val _ = textLeft := tl
         val _ = textFinishFn := tf
      in
         ()
      end
   fun finishText textRight =
      let
         val cs = String.fromListRev (!chars)
         val tl = !textLeft
         val tr = textRight
         val tf = !textFinishFn
         val _ = chars := []
         val _ = inText := false
         val _ = textLeft := SourcePos.bogus
         val _ = textFinishFn := (fn _ => raise Fail "textFinish")
      in
         tf (cs, tl, tr)
      end
   fun addTextString (s: string) =
      chars := String.fold (s, !chars, fn (c, ac) => c :: ac)
   val inText = fn () => !inText
end
fun addTextChar (c: char) = addTextString (String.fromChar c)
fun addTextNumEsc (source, yypos, yytext, drop, radix): unit =
   let
      val left = yypos
      val right = lastPos (left, yytext)
      fun err () =
         error (source, left, right, "Illegal numeric escape in text constant")
   in
      case StringCvt.scanString (fn r => IntInf.scan (radix, r)) (String.dropPrefix (yytext, drop)) of
         NONE => err ()
       | SOME i => if i > 255
                      then err ()
                      else addTextChar (Char.chr (IntInf.toInt i))
   end
fun addTextUTF8 (source, yypos, yytext): unit =
   addTextString yytext


(* EOF *)
val eof: lexarg -> lexresult =
   fn {source, ...} =>
   let
      val _ = Source.newline (source, ~1)
      val pos = Source.getPos (source, ~1)
      val _ =
         if inComment ()
            then error' (pos, SourcePos.bogus, "Unclosed comment at end of file")
            else ()
      val _ =
         if inText ()
            then error' (pos, SourcePos.bogus, "Unclosed text constant at end of file")
            else ()
   in
      Tokens.EOF (pos, SourcePos.bogus)
   end


(*#line 178.1 "mlb.lex.sml"*)
end (* end of user routines *)
exception LexError (* raised if illegal leaf action tried *)
structure Internal =
	struct

datatype yyfinstate = N of int
type statedata = {fin : yyfinstate list, trans: string}
(* transition & final state table *)
val tab = let
val s = [ 
 (0, 
"\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000"
),
 (1, 
"\019\019\019\019\019\019\019\019\019\112\115\112\112\114\019\019\
\\019\019\019\019\019\019\019\019\019\019\019\019\019\019\019\019\
\\112\019\111\019\110\019\019\019\101\019\019\019\100\019\099\089\
\\088\088\088\088\088\088\088\088\088\088\019\087\019\086\019\019\
\\019\020\020\020\020\020\020\020\020\020\020\020\020\020\020\020\
\\020\020\020\020\020\020\020\020\020\020\020\019\019\019\019\081\
\\019\077\072\020\020\069\062\020\020\060\020\020\053\020\020\049\
\\020\020\020\032\020\020\020\020\020\020\020\019\019\019\019\019\
\\019\019\019\019\019\019\019\019\019\019\019\019\019\019\019\019\
\\019\019\019\019\019\019\019\019\019\019\019\019\019\019\019\019\
\\019\019\019\019\019\019\019\019\019\019\019\019\019\019\019\019\
\\019\019\019\019\019\019\019\019\019\019\019\019\019\019\019\019\
\\019\019\019\019\019\019\019\019\019\019\019\019\019\019\019\019\
\\019\019\019\019\019\019\019\019\019\019\019\019\019\019\019\019\
\\019\019\019\019\019\019\019\019\019\019\019\019\019\019\019\019\
\\019\019\019\019\019\019\019\019\019\019\019\019\019\019\019\019"
),
 (3, 
"\116\116\116\116\116\116\116\116\116\116\162\116\116\161\116\116\
\\116\116\116\116\116\116\116\116\116\116\116\116\116\116\116\116\
\\126\126\160\126\126\126\126\126\126\126\126\126\126\126\126\126\
\\126\126\126\126\126\126\126\126\126\126\126\126\126\126\126\126\
\\126\126\126\126\126\126\126\126\126\126\126\126\126\126\126\126\
\\126\126\126\126\126\126\126\126\126\126\126\126\127\126\126\126\
\\126\126\126\126\126\126\126\126\126\126\126\126\126\126\126\126\
\\126\126\126\126\126\126\126\126\126\126\126\126\126\126\126\116\
\\116\116\116\116\116\116\116\116\116\116\116\116\116\116\116\116\
\\116\116\116\116\116\116\116\116\116\116\116\116\116\116\116\116\
\\116\116\116\116\116\116\116\116\116\116\116\116\116\116\116\116\
\\116\116\116\116\116\116\116\116\116\116\116\116\116\116\116\116\
\\124\124\124\124\124\124\124\124\124\124\124\124\124\124\124\124\
\\124\124\124\124\124\124\124\124\124\124\124\124\124\124\124\124\
\\121\121\121\121\121\121\121\121\121\121\121\121\121\121\121\121\
\\117\117\117\117\117\117\117\117\116\116\116\116\116\116\116\116"
),
 (5, 
"\163\163\163\163\163\163\163\163\163\165\168\165\165\167\163\163\
\\163\163\163\163\163\163\163\163\163\163\163\163\163\163\163\163\
\\165\163\163\163\163\163\163\163\163\163\163\163\163\163\163\163\
\\163\163\163\163\163\163\163\163\163\163\163\163\163\163\163\163\
\\163\163\163\163\163\163\163\163\163\163\163\163\163\163\163\163\
\\163\163\163\163\163\163\163\163\163\163\163\163\164\163\163\163\
\\163\163\163\163\163\163\163\163\163\163\163\163\163\163\163\163\
\\163\163\163\163\163\163\163\163\163\163\163\163\163\163\163\163\
\\163\163\163\163\163\163\163\163\163\163\163\163\163\163\163\163\
\\163\163\163\163\163\163\163\163\163\163\163\163\163\163\163\163\
\\163\163\163\163\163\163\163\163\163\163\163\163\163\163\163\163\
\\163\163\163\163\163\163\163\163\163\163\163\163\163\163\163\163\
\\163\163\163\163\163\163\163\163\163\163\163\163\163\163\163\163\
\\163\163\163\163\163\163\163\163\163\163\163\163\163\163\163\163\
\\163\163\163\163\163\163\163\163\163\163\163\163\163\163\163\163\
\\163\163\163\163\163\163\163\163\163\163\163\163\163\163\163\163"
),
 (7, 
"\169\169\169\169\169\169\169\169\169\169\176\169\169\175\169\169\
\\169\169\169\169\169\169\169\169\169\169\169\169\169\169\169\169\
\\169\169\169\169\169\169\169\169\172\169\170\169\169\169\169\169\
\\169\169\169\169\169\169\169\169\169\169\169\169\169\169\169\169\
\\169\169\169\169\169\169\169\169\169\169\169\169\169\169\169\169\
\\169\169\169\169\169\169\169\169\169\169\169\169\169\169\169\169\
\\169\169\169\169\169\169\169\169\169\169\169\169\169\169\169\169\
\\169\169\169\169\169\169\169\169\169\169\169\169\169\169\169\169\
\\169\169\169\169\169\169\169\169\169\169\169\169\169\169\169\169\
\\169\169\169\169\169\169\169\169\169\169\169\169\169\169\169\169\
\\169\169\169\169\169\169\169\169\169\169\169\169\169\169\169\169\
\\169\169\169\169\169\169\169\169\169\169\169\169\169\169\169\169\
\\169\169\169\169\169\169\169\169\169\169\169\169\169\169\169\169\
\\169\169\169\169\169\169\169\169\169\169\169\169\169\169\169\169\
\\169\169\169\169\169\169\169\169\169\169\169\169\169\169\169\169\
\\169\169\169\169\169\169\169\169\169\169\169\169\169\169\169\169"
),
 (9, 
"\177\177\177\177\177\177\177\177\177\177\179\177\177\178\177\177\
\\177\177\177\177\177\177\177\177\177\177\177\177\177\177\177\177\
\\177\177\177\177\177\177\177\177\177\177\177\177\177\177\177\177\
\\177\177\177\177\177\177\177\177\177\177\177\177\177\177\177\177\
\\177\177\177\177\177\177\177\177\177\177\177\177\177\177\177\177\
\\177\177\177\177\177\177\177\177\177\177\177\177\177\177\177\177\
\\177\177\177\177\177\177\177\177\177\177\177\177\177\177\177\177\
\\177\177\177\177\177\177\177\177\177\177\177\177\177\177\177\177\
\\177\177\177\177\177\177\177\177\177\177\177\177\177\177\177\177\
\\177\177\177\177\177\177\177\177\177\177\177\177\177\177\177\177\
\\177\177\177\177\177\177\177\177\177\177\177\177\177\177\177\177\
\\177\177\177\177\177\177\177\177\177\177\177\177\177\177\177\177\
\\177\177\177\177\177\177\177\177\177\177\177\177\177\177\177\177\
\\177\177\177\177\177\177\177\177\177\177\177\177\177\177\177\177\
\\177\177\177\177\177\177\177\177\177\177\177\177\177\177\177\177\
\\177\177\177\177\177\177\177\177\177\177\177\177\177\177\177\177"
),
 (11, 
"\180\180\180\180\180\180\180\180\180\180\000\180\180\180\180\180\
\\180\180\180\180\180\180\180\180\180\180\180\180\180\180\180\180\
\\180\180\180\180\180\180\180\180\180\180\180\180\180\180\180\180\
\\181\181\181\181\181\181\181\181\181\181\180\180\180\180\180\180\
\\180\180\180\180\180\180\180\180\180\180\180\180\180\180\180\180\
\\180\180\180\180\180\180\180\180\180\180\180\180\180\180\180\180\
\\180\180\180\180\180\180\180\180\180\180\180\180\180\180\180\180\
\\180\180\180\180\180\180\180\180\180\180\180\180\180\180\180\180\
\\180\180\180\180\180\180\180\180\180\180\180\180\180\180\180\180\
\\180\180\180\180\180\180\180\180\180\180\180\180\180\180\180\180\
\\180\180\180\180\180\180\180\180\180\180\180\180\180\180\180\180\
\\180\180\180\180\180\180\180\180\180\180\180\180\180\180\180\180\
\\180\180\180\180\180\180\180\180\180\180\180\180\180\180\180\180\
\\180\180\180\180\180\180\180\180\180\180\180\180\180\180\180\180\
\\180\180\180\180\180\180\180\180\180\180\180\180\180\180\180\180\
\\180\180\180\180\180\180\180\180\180\180\180\180\180\180\180\180"
),
 (13, 
"\180\180\180\180\180\180\180\180\180\187\000\187\187\180\180\180\
\\180\180\180\180\180\180\180\180\180\180\180\180\180\180\180\180\
\\187\180\180\180\180\180\180\180\180\180\185\180\180\180\180\180\
\\180\180\180\180\180\180\180\180\180\180\180\180\180\180\180\180\
\\180\180\180\180\180\180\180\180\180\180\180\180\180\180\180\180\
\\180\180\180\180\180\180\180\180\180\180\180\180\180\180\180\180\
\\180\180\180\180\180\180\180\180\180\180\180\180\180\180\180\180\
\\180\180\180\180\180\180\180\180\180\180\180\180\180\180\180\180\
\\180\180\180\180\180\180\180\180\180\180\180\180\180\180\180\180\
\\180\180\180\180\180\180\180\180\180\180\180\180\180\180\180\180\
\\180\180\180\180\180\180\180\180\180\180\180\180\180\180\180\180\
\\180\180\180\180\180\180\180\180\180\180\180\180\180\180\180\180\
\\180\180\180\180\180\180\180\180\180\180\180\180\180\180\180\180\
\\180\180\180\180\180\180\180\180\180\180\180\180\180\180\180\180\
\\180\180\180\180\180\180\180\180\180\180\180\180\180\180\180\180\
\\180\180\180\180\180\180\180\180\180\180\180\180\180\180\180\180"
),
 (15, 
"\191\191\191\191\191\191\191\191\191\191\192\191\191\191\191\191\
\\191\191\191\191\191\191\191\191\191\191\191\191\191\191\191\191\
\\191\191\194\191\191\191\191\191\191\191\191\191\191\191\191\191\
\\191\191\191\191\191\191\191\191\191\191\191\191\191\191\191\191\
\\191\191\191\191\191\191\191\191\191\191\191\191\191\191\191\191\
\\191\191\191\191\191\191\191\191\191\191\191\191\191\191\191\191\
\\191\191\191\191\191\191\191\191\191\191\191\191\191\191\191\191\
\\191\191\191\191\191\191\191\191\191\191\191\191\191\191\191\191\
\\191\191\191\191\191\191\191\191\191\191\191\191\191\191\191\191\
\\191\191\191\191\191\191\191\191\191\191\191\191\191\191\191\191\
\\191\191\191\191\191\191\191\191\191\191\191\191\191\191\191\191\
\\191\191\191\191\191\191\191\191\191\191\191\191\191\191\191\191\
\\191\191\191\191\191\191\191\191\191\191\191\191\191\191\191\191\
\\191\191\191\191\191\191\191\191\191\191\191\191\191\191\191\191\
\\191\191\191\191\191\191\191\191\191\191\191\191\191\191\191\191\
\\191\191\191\191\191\191\191\191\191\191\191\191\191\191\191\191"
),
 (17, 
"\180\180\180\180\180\180\180\180\180\195\000\195\195\180\180\180\
\\180\180\180\180\180\180\180\180\180\180\180\180\180\180\180\180\
\\195\180\180\180\180\180\180\180\180\180\185\180\180\180\180\180\
\\180\180\180\180\180\180\180\180\180\180\180\180\180\180\180\180\
\\180\180\180\180\180\180\180\180\180\180\180\180\180\180\180\180\
\\180\180\180\180\180\180\180\180\180\180\180\180\180\180\180\180\
\\180\180\180\180\180\180\180\180\180\180\180\180\180\180\180\180\
\\180\180\180\180\180\180\180\180\180\180\180\180\180\180\180\180\
\\180\180\180\180\180\180\180\180\180\180\180\180\180\180\180\180\
\\180\180\180\180\180\180\180\180\180\180\180\180\180\180\180\180\
\\180\180\180\180\180\180\180\180\180\180\180\180\180\180\180\180\
\\180\180\180\180\180\180\180\180\180\180\180\180\180\180\180\180\
\\180\180\180\180\180\180\180\180\180\180\180\180\180\180\180\180\
\\180\180\180\180\180\180\180\180\180\180\180\180\180\180\180\180\
\\180\180\180\180\180\180\180\180\180\180\180\180\180\180\180\180\
\\180\180\180\180\180\180\180\180\180\180\180\180\180\180\180\180"
),
 (20, 
"\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\\000\000\000\000\024\000\000\031\000\000\000\000\000\023\023\022\
\\021\021\021\021\021\021\021\021\021\021\000\000\000\000\000\000\
\\000\021\021\021\021\021\021\021\021\021\021\021\021\021\021\021\
\\021\021\021\021\021\021\021\021\021\021\021\000\000\000\000\021\
\\000\021\021\021\021\021\021\021\021\021\021\021\021\021\021\021\
\\021\021\021\021\021\021\021\021\021\021\021\000\000\000\000\000\
\\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000"
),
 (22, 
"\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\\000\000\000\000\028\000\000\000\000\000\000\000\000\000\027\000\
\\023\023\023\023\023\023\023\023\023\023\000\000\000\000\000\000\
\\000\023\023\023\023\023\023\023\023\023\023\023\023\023\023\023\
\\023\023\023\023\023\023\023\023\023\023\023\000\000\000\000\023\
\\000\023\023\023\023\023\023\023\023\023\023\023\023\023\023\023\
\\023\023\023\023\023\023\023\023\023\023\023\000\000\000\000\000\
\\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000"
),
 (23, 
"\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\\000\000\000\000\024\000\000\000\000\000\000\000\000\023\023\022\
\\023\023\023\023\023\023\023\023\023\023\000\000\000\000\000\000\
\\000\023\023\023\023\023\023\023\023\023\023\023\023\023\023\023\
\\023\023\023\023\023\023\023\023\023\023\023\000\000\000\000\023\
\\000\023\023\023\023\023\023\023\023\023\023\023\023\023\023\023\
\\023\023\023\023\023\023\023\023\023\023\023\000\000\000\000\000\
\\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000"
),
 (24, 
"\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\\000\000\000\000\000\000\000\000\025\000\000\000\000\000\000\000\
\\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000"
),
 (25, 
"\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\\000\026\026\026\026\026\026\026\026\026\026\026\026\026\026\026\
\\026\026\026\026\026\026\026\026\026\026\026\000\000\000\000\026\
\\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000"
),
 (26, 
"\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\\000\000\000\000\000\000\000\000\000\023\000\000\000\000\000\000\
\\026\026\026\026\026\026\026\026\026\026\000\000\000\000\000\000\
\\000\026\026\026\026\026\026\026\026\026\026\026\026\026\026\026\
\\026\026\026\026\026\026\026\026\026\026\026\000\000\000\000\026\
\\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000"
),
 (28, 
"\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\\000\000\000\000\000\000\000\000\029\000\000\000\000\000\000\000\
\\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000"
),
 (29, 
"\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\\000\030\030\030\030\030\030\030\030\030\030\030\030\030\030\030\
\\030\030\030\030\030\030\030\030\030\030\030\000\000\000\000\030\
\\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000"
),
 (30, 
"\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\\000\000\000\000\000\000\000\000\000\023\000\000\000\000\000\000\
\\030\030\030\030\030\030\030\030\030\030\000\000\000\000\000\000\
\\000\030\030\030\030\030\030\030\030\030\030\030\030\030\030\030\
\\030\030\030\030\030\030\030\030\030\030\030\000\000\000\000\030\
\\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000"
),
 (31, 
"\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\\000\000\000\000\000\000\000\031\000\000\000\000\000\000\000\000\
\\031\031\031\031\031\031\031\031\031\031\000\000\000\000\000\000\
\\000\031\031\031\031\031\031\031\031\031\031\031\031\031\031\031\
\\031\031\031\031\031\031\031\031\031\031\031\000\000\000\000\031\
\\000\031\031\031\031\031\031\031\031\031\031\031\031\031\031\031\
\\031\031\031\031\031\031\031\031\031\031\031\000\000\000\000\000\
\\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000"
),
 (32, 
"\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\\000\000\000\000\024\000\000\031\000\000\000\000\000\023\023\022\
\\021\021\021\021\021\021\021\021\021\021\000\000\000\000\000\000\
\\000\021\021\021\021\021\021\021\021\021\021\021\021\021\021\021\
\\021\021\021\021\021\021\021\021\021\021\021\000\000\000\000\021\
\\000\021\021\021\021\021\021\021\021\041\021\021\021\021\021\021\
\\021\021\021\021\033\021\021\021\021\021\021\000\000\000\000\000\
\\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000"
),
 (33, 
"\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\\000\000\000\000\024\000\000\031\000\000\000\000\000\023\023\022\
\\021\021\021\021\021\021\021\021\021\021\000\000\000\000\000\000\
\\000\021\021\021\021\021\021\021\021\021\021\021\021\021\021\021\
\\021\021\021\021\021\021\021\021\021\021\021\000\000\000\000\021\
\\000\021\021\021\021\021\021\021\021\021\021\021\021\021\021\021\
\\021\021\034\021\021\021\021\021\021\021\021\000\000\000\000\000\
\\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000"
),
 (34, 
"\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\\000\000\000\000\024\000\000\031\000\000\000\000\000\023\023\022\
\\021\021\021\021\021\021\021\021\021\021\000\000\000\000\000\000\
\\000\021\021\021\021\021\021\021\021\021\021\021\021\021\021\021\
\\021\021\021\021\021\021\021\021\021\021\021\000\000\000\000\021\
\\000\021\021\021\021\021\021\021\021\021\021\021\021\021\021\021\
\\021\021\021\021\021\035\021\021\021\021\021\000\000\000\000\000\
\\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000"
),
 (35, 
"\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\\000\000\000\000\024\000\000\031\000\000\000\000\000\023\023\022\
\\021\021\021\021\021\021\021\021\021\021\000\000\000\000\000\000\
\\000\021\021\021\021\021\021\021\021\021\021\021\021\021\021\021\
\\021\021\021\021\021\021\021\021\021\021\021\000\000\000\000\021\
\\000\021\021\036\021\021\021\021\021\021\021\021\021\021\021\021\
\\021\021\021\021\021\021\021\021\021\021\021\000\000\000\000\000\
\\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000"
),
 (36, 
"\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\\000\000\000\000\024\000\000\031\000\000\000\000\000\023\023\022\
\\021\021\021\021\021\021\021\021\021\021\000\000\000\000\000\000\
\\000\021\021\021\021\021\021\021\021\021\021\021\021\021\021\021\
\\021\021\021\021\021\021\021\021\021\021\021\000\000\000\000\021\
\\000\021\021\021\021\021\021\021\021\021\021\021\021\021\021\021\
\\021\021\021\021\037\021\021\021\021\021\021\000\000\000\000\000\
\\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000"
),
 (37, 
"\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\\000\000\000\000\024\000\000\031\000\000\000\000\000\023\023\022\
\\021\021\021\021\021\021\021\021\021\021\000\000\000\000\000\000\
\\000\021\021\021\021\021\021\021\021\021\021\021\021\021\021\021\
\\021\021\021\021\021\021\021\021\021\021\021\000\000\000\000\021\
\\000\021\021\021\021\021\021\021\021\021\021\021\021\021\021\021\
\\021\021\021\021\021\038\021\021\021\021\021\000\000\000\000\000\
\\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000"
),
 (38, 
"\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\\000\000\000\000\024\000\000\031\000\000\000\000\000\023\023\022\
\\021\021\021\021\021\021\021\021\021\021\000\000\000\000\000\000\
\\000\021\021\021\021\021\021\021\021\021\021\021\021\021\021\021\
\\021\021\021\021\021\021\021\021\021\021\021\000\000\000\000\021\
\\000\021\021\021\021\021\021\021\021\021\021\021\021\021\021\021\
\\021\021\039\021\021\021\021\021\021\021\021\000\000\000\000\000\
\\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000"
),
 (39, 
"\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\\000\000\000\000\024\000\000\031\000\000\000\000\000\023\023\022\
\\021\021\021\021\021\021\021\021\021\021\000\000\000\000\000\000\
\\000\021\021\021\021\021\021\021\021\021\021\021\021\021\021\021\
\\021\021\021\021\021\021\021\021\021\021\021\000\000\000\000\021\
\\000\021\021\021\021\040\021\021\021\021\021\021\021\021\021\021\
\\021\021\021\021\021\021\021\021\021\021\021\000\000\000\000\000\
\\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000"
),
 (41, 
"\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\\000\000\000\000\024\000\000\031\000\000\000\000\000\023\023\022\
\\021\021\021\021\021\021\021\021\021\021\000\000\000\000\000\000\
\\000\021\021\021\021\021\021\021\021\021\021\021\021\021\021\021\
\\021\021\021\021\021\021\021\021\021\021\021\000\000\000\000\021\
\\000\021\021\021\021\021\021\042\021\021\021\021\021\021\021\021\
\\021\021\021\021\021\021\021\021\021\021\021\000\000\000\000\000\
\\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000"
),
 (42, 
"\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\\000\000\000\000\024\000\000\031\000\000\000\000\000\023\023\022\
\\021\021\021\021\021\021\021\021\021\021\000\000\000\000\000\000\
\\000\021\021\021\021\021\021\021\021\021\021\021\021\021\021\021\
\\021\021\021\021\021\021\021\021\021\021\021\000\000\000\000\021\
\\000\021\021\021\021\021\021\021\021\021\021\021\021\021\043\021\
\\021\021\021\021\021\021\021\021\021\021\021\000\000\000\000\000\
\\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000"
),
 (43, 
"\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\\000\000\000\000\024\000\000\031\000\000\000\000\000\023\023\022\
\\021\021\021\021\021\021\021\021\021\021\000\000\000\000\000\000\
\\000\021\021\021\021\021\021\021\021\021\021\021\021\021\021\021\
\\021\021\021\021\021\021\021\021\021\021\021\000\000\000\000\021\
\\000\044\021\021\021\021\021\021\021\021\021\021\021\021\021\021\
\\021\021\021\021\021\021\021\021\021\021\021\000\000\000\000\000\
\\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000"
),
 (44, 
"\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\\000\000\000\000\024\000\000\031\000\000\000\000\000\023\023\022\
\\021\021\021\021\021\021\021\021\021\021\000\000\000\000\000\000\
\\000\021\021\021\021\021\021\021\021\021\021\021\021\021\021\021\
\\021\021\021\021\021\021\021\021\021\021\021\000\000\000\000\021\
\\000\021\021\021\021\021\021\021\021\021\021\021\021\021\021\021\
\\021\021\021\021\045\021\021\021\021\021\021\000\000\000\000\000\
\\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000"
),
 (45, 
"\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\\000\000\000\000\024\000\000\031\000\000\000\000\000\023\023\022\
\\021\021\021\021\021\021\021\021\021\021\000\000\000\000\000\000\
\\000\021\021\021\021\021\021\021\021\021\021\021\021\021\021\021\
\\021\021\021\021\021\021\021\021\021\021\021\000\000\000\000\021\
\\000\021\021\021\021\021\021\021\021\021\021\021\021\021\021\021\
\\021\021\021\021\021\046\021\021\021\021\021\000\000\000\000\000\
\\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000"
),
 (46, 
"\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\\000\000\000\000\024\000\000\031\000\000\000\000\000\023\023\022\
\\021\021\021\021\021\021\021\021\021\021\000\000\000\000\000\000\
\\000\021\021\021\021\021\021\021\021\021\021\021\021\021\021\021\
\\021\021\021\021\021\021\021\021\021\021\021\000\000\000\000\021\
\\000\021\021\021\021\021\021\021\021\021\021\021\021\021\021\021\
\\021\021\047\021\021\021\021\021\021\021\021\000\000\000\000\000\
\\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000"
),
 (47, 
"\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\\000\000\000\000\024\000\000\031\000\000\000\000\000\023\023\022\
\\021\021\021\021\021\021\021\021\021\021\000\000\000\000\000\000\
\\000\021\021\021\021\021\021\021\021\021\021\021\021\021\021\021\
\\021\021\021\021\021\021\021\021\021\021\021\000\000\000\000\021\
\\000\021\021\021\021\048\021\021\021\021\021\021\021\021\021\021\
\\021\021\021\021\021\021\021\021\021\021\021\000\000\000\000\000\
\\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000"
),
 (49, 
"\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\\000\000\000\000\024\000\000\031\000\000\000\000\000\023\023\022\
\\021\021\021\021\021\021\021\021\021\021\000\000\000\000\000\000\
\\000\021\021\021\021\021\021\021\021\021\021\021\021\021\021\021\
\\021\021\021\021\021\021\021\021\021\021\021\000\000\000\000\021\
\\000\021\021\021\021\021\021\021\021\021\021\021\021\021\021\021\
\\050\021\021\021\021\021\021\021\021\021\021\000\000\000\000\000\
\\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000"
),
 (50, 
"\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\\000\000\000\000\024\000\000\031\000\000\000\000\000\023\023\022\
\\021\021\021\021\021\021\021\021\021\021\000\000\000\000\000\000\
\\000\021\021\021\021\021\021\021\021\021\021\021\021\021\021\021\
\\021\021\021\021\021\021\021\021\021\021\021\000\000\000\000\021\
\\000\021\021\021\021\051\021\021\021\021\021\021\021\021\021\021\
\\021\021\021\021\021\021\021\021\021\021\021\000\000\000\000\000\
\\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000"
),
 (51, 
"\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\\000\000\000\000\024\000\000\031\000\000\000\000\000\023\023\022\
\\021\021\021\021\021\021\021\021\021\021\000\000\000\000\000\000\
\\000\021\021\021\021\021\021\021\021\021\021\021\021\021\021\021\
\\021\021\021\021\021\021\021\021\021\021\021\000\000\000\000\021\
\\000\021\021\021\021\021\021\021\021\021\021\021\021\021\052\021\
\\021\021\021\021\021\021\021\021\021\021\021\000\000\000\000\000\
\\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000"
),
 (53, 
"\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\\000\000\000\000\024\000\000\031\000\000\000\000\000\023\023\022\
\\021\021\021\021\021\021\021\021\021\021\000\000\000\000\000\000\
\\000\021\021\021\021\021\021\021\021\021\021\021\021\021\021\021\
\\021\021\021\021\021\021\021\021\021\021\021\000\000\000\000\021\
\\000\021\021\021\021\058\021\021\021\021\021\021\021\021\021\054\
\\021\021\021\021\021\021\021\021\021\021\021\000\000\000\000\000\
\\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000"
),
 (54, 
"\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\\000\000\000\000\024\000\000\031\000\000\000\000\000\023\023\022\
\\021\021\021\021\021\021\021\021\021\021\000\000\000\000\000\000\
\\000\021\021\021\021\021\021\021\021\021\021\021\021\021\021\021\
\\021\021\021\021\021\021\021\021\021\021\021\000\000\000\000\021\
\\000\021\021\055\021\021\021\021\021\021\021\021\021\021\021\021\
\\021\021\021\021\021\021\021\021\021\021\021\000\000\000\000\000\
\\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000"
),
 (55, 
"\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\\000\000\000\000\024\000\000\031\000\000\000\000\000\023\023\022\
\\021\021\021\021\021\021\021\021\021\021\000\000\000\000\000\000\
\\000\021\021\021\021\021\021\021\021\021\021\021\021\021\021\021\
\\021\021\021\021\021\021\021\021\021\021\021\000\000\000\000\021\
\\000\056\021\021\021\021\021\021\021\021\021\021\021\021\021\021\
\\021\021\021\021\021\021\021\021\021\021\021\000\000\000\000\000\
\\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000"
),
 (56, 
"\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\\000\000\000\000\024\000\000\031\000\000\000\000\000\023\023\022\
\\021\021\021\021\021\021\021\021\021\021\000\000\000\000\000\000\
\\000\021\021\021\021\021\021\021\021\021\021\021\021\021\021\021\
\\021\021\021\021\021\021\021\021\021\021\021\000\000\000\000\021\
\\000\021\021\021\021\021\021\021\021\021\021\021\057\021\021\021\
\\021\021\021\021\021\021\021\021\021\021\021\000\000\000\000\000\
\\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000"
),
 (58, 
"\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\\000\000\000\000\024\000\000\031\000\000\000\000\000\023\023\022\
\\021\021\021\021\021\021\021\021\021\021\000\000\000\000\000\000\
\\000\021\021\021\021\021\021\021\021\021\021\021\021\021\021\021\
\\021\021\021\021\021\021\021\021\021\021\021\000\000\000\000\021\
\\000\021\021\021\021\021\021\021\021\021\021\021\021\021\021\021\
\\021\021\021\021\059\021\021\021\021\021\021\000\000\000\000\000\
\\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000"
),
 (60, 
"\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\\000\000\000\000\024\000\000\031\000\000\000\000\000\023\023\022\
\\021\021\021\021\021\021\021\021\021\021\000\000\000\000\000\000\
\\000\021\021\021\021\021\021\021\021\021\021\021\021\021\021\021\
\\021\021\021\021\021\021\021\021\021\021\021\000\000\000\000\021\
\\000\021\021\021\021\021\021\021\021\021\021\021\021\021\061\021\
\\021\021\021\021\021\021\021\021\021\021\021\000\000\000\000\000\
\\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000"
),
 (62, 
"\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\\000\000\000\000\024\000\000\031\000\000\000\000\000\023\023\022\
\\021\021\021\021\021\021\021\021\021\021\000\000\000\000\000\000\
\\000\021\021\021\021\021\021\021\021\021\021\021\021\021\021\021\
\\021\021\021\021\021\021\021\021\021\021\021\000\000\000\000\021\
\\000\021\021\021\021\021\021\021\021\021\021\021\021\021\021\021\
\\021\021\021\021\021\063\021\021\021\021\021\000\000\000\000\000\
\\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000"
),
 (63, 
"\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\\000\000\000\000\024\000\000\031\000\000\000\000\000\023\023\022\
\\021\021\021\021\021\021\021\021\021\021\000\000\000\000\000\000\
\\000\021\021\021\021\021\021\021\021\021\021\021\021\021\021\021\
\\021\021\021\021\021\021\021\021\021\021\021\000\000\000\000\021\
\\000\021\021\021\021\021\021\021\021\021\021\021\021\021\064\021\
\\021\021\021\021\021\021\021\021\021\021\021\000\000\000\000\000\
\\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000"
),
 (64, 
"\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\\000\000\000\000\024\000\000\031\000\000\000\000\000\023\023\022\
\\021\021\021\021\021\021\021\021\021\021\000\000\000\000\000\000\
\\000\021\021\021\021\021\021\021\021\021\021\021\021\021\021\021\
\\021\021\021\021\021\021\021\021\021\021\021\000\000\000\000\021\
\\000\021\021\065\021\021\021\021\021\021\021\021\021\021\021\021\
\\021\021\021\021\021\021\021\021\021\021\021\000\000\000\000\000\
\\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000"
),
 (65, 
"\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\\000\000\000\000\024\000\000\031\000\000\000\000\000\023\023\022\
\\021\021\021\021\021\021\021\021\021\021\000\000\000\000\000\000\
\\000\021\021\021\021\021\021\021\021\021\021\021\021\021\021\021\
\\021\021\021\021\021\021\021\021\021\021\021\000\000\000\000\021\
\\000\021\021\021\021\021\021\021\021\021\021\021\021\021\021\021\
\\021\021\021\021\066\021\021\021\021\021\021\000\000\000\000\000\
\\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000"
),
 (66, 
"\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\\000\000\000\000\024\000\000\031\000\000\000\000\000\023\023\022\
\\021\021\021\021\021\021\021\021\021\021\000\000\000\000\000\000\
\\000\021\021\021\021\021\021\021\021\021\021\021\021\021\021\021\
\\021\021\021\021\021\021\021\021\021\021\021\000\000\000\000\021\
\\000\021\021\021\021\021\021\021\021\021\021\021\021\021\021\067\
\\021\021\021\021\021\021\021\021\021\021\021\000\000\000\000\000\
\\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000"
),
 (67, 
"\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\\000\000\000\000\024\000\000\031\000\000\000\000\000\023\023\022\
\\021\021\021\021\021\021\021\021\021\021\000\000\000\000\000\000\
\\000\021\021\021\021\021\021\021\021\021\021\021\021\021\021\021\
\\021\021\021\021\021\021\021\021\021\021\021\000\000\000\000\021\
\\000\021\021\021\021\021\021\021\021\021\021\021\021\021\021\021\
\\021\021\068\021\021\021\021\021\021\021\021\000\000\000\000\000\
\\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000"
),
 (69, 
"\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\\000\000\000\000\024\000\000\031\000\000\000\000\000\023\023\022\
\\021\021\021\021\021\021\021\021\021\021\000\000\000\000\000\000\
\\000\021\021\021\021\021\021\021\021\021\021\021\021\021\021\021\
\\021\021\021\021\021\021\021\021\021\021\021\000\000\000\000\021\
\\000\021\021\021\021\021\021\021\021\021\021\021\021\021\070\021\
\\021\021\021\021\021\021\021\021\021\021\021\000\000\000\000\000\
\\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000"
),
 (70, 
"\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\\000\000\000\000\024\000\000\031\000\000\000\000\000\023\023\022\
\\021\021\021\021\021\021\021\021\021\021\000\000\000\000\000\000\
\\000\021\021\021\021\021\021\021\021\021\021\021\021\021\021\021\
\\021\021\021\021\021\021\021\021\021\021\021\000\000\000\000\021\
\\000\021\021\021\071\021\021\021\021\021\021\021\021\021\021\021\
\\021\021\021\021\021\021\021\021\021\021\021\000\000\000\000\000\
\\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000"
),
 (72, 
"\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\\000\000\000\000\024\000\000\031\000\000\000\000\000\023\023\022\
\\021\021\021\021\021\021\021\021\021\021\000\000\000\000\000\000\
\\000\021\021\021\021\021\021\021\021\021\021\021\021\021\021\021\
\\021\021\021\021\021\021\021\021\021\021\021\000\000\000\000\021\
\\000\073\021\021\021\021\021\021\021\021\021\021\021\021\021\021\
\\021\021\021\021\021\021\021\021\021\021\021\000\000\000\000\000\
\\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000"
),
 (73, 
"\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\\000\000\000\000\024\000\000\031\000\000\000\000\000\023\023\022\
\\021\021\021\021\021\021\021\021\021\021\000\000\000\000\000\000\
\\000\021\021\021\021\021\021\021\021\021\021\021\021\021\021\021\
\\021\021\021\021\021\021\021\021\021\021\021\000\000\000\000\021\
\\000\021\021\021\021\021\021\021\021\021\021\021\021\021\021\021\
\\021\021\021\074\021\021\021\021\021\021\021\000\000\000\000\000\
\\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000"
),
 (74, 
"\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\\000\000\000\000\024\000\000\031\000\000\000\000\000\023\023\022\
\\021\021\021\021\021\021\021\021\021\021\000\000\000\000\000\000\
\\000\021\021\021\021\021\021\021\021\021\021\021\021\021\021\021\
\\021\021\021\021\021\021\021\021\021\021\021\000\000\000\000\021\
\\000\021\021\021\021\021\021\021\021\075\021\021\021\021\021\021\
\\021\021\021\021\021\021\021\021\021\021\021\000\000\000\000\000\
\\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000"
),
 (75, 
"\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\\000\000\000\000\024\000\000\031\000\000\000\000\000\023\023\022\
\\021\021\021\021\021\021\021\021\021\021\000\000\000\000\000\000\
\\000\021\021\021\021\021\021\021\021\021\021\021\021\021\021\021\
\\021\021\021\021\021\021\021\021\021\021\021\000\000\000\000\021\
\\000\021\021\021\021\021\021\021\021\021\021\021\021\021\021\021\
\\021\021\021\076\021\021\021\021\021\021\021\000\000\000\000\000\
\\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000"
),
 (77, 
"\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\\000\000\000\000\024\000\000\031\000\000\000\000\000\023\023\022\
\\021\021\021\021\021\021\021\021\021\021\000\000\000\000\000\000\
\\000\021\021\021\021\021\021\021\021\021\021\021\021\021\021\021\
\\021\021\021\021\021\021\021\021\021\021\021\000\000\000\000\021\
\\000\021\021\021\021\021\021\021\021\021\021\021\021\021\078\021\
\\021\021\021\021\021\021\021\021\021\021\021\000\000\000\000\000\
\\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000"
),
 (78, 
"\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\\000\000\000\000\024\000\000\031\000\000\000\000\000\023\023\022\
\\021\021\021\021\021\021\021\021\021\021\000\000\000\000\000\000\
\\000\021\021\021\021\021\021\021\021\021\021\021\021\021\021\021\
\\021\021\021\021\021\021\021\021\021\021\021\000\000\000\000\021\
\\000\021\021\021\080\021\021\021\021\021\021\021\021\021\079\021\
\\021\021\021\021\021\021\021\021\021\021\021\000\000\000\000\000\
\\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000"
),
 (81, 
"\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\\000\000\000\000\024\000\000\000\000\000\000\000\000\023\023\022\
\\023\023\023\023\023\023\023\023\023\023\000\000\000\000\000\000\
\\000\023\023\023\023\023\023\023\023\023\023\023\023\023\023\023\
\\023\023\023\023\023\023\023\023\023\023\023\000\000\000\000\023\
\\000\023\023\023\023\023\023\023\023\023\023\023\023\023\023\023\
\\082\023\023\023\023\023\023\023\023\023\023\000\000\000\000\000\
\\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000"
),
 (82, 
"\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\\000\000\000\000\024\000\000\000\000\000\000\000\000\023\023\022\
\\023\023\023\023\023\023\023\023\023\023\000\000\000\000\000\000\
\\000\023\023\023\023\023\023\023\023\023\023\023\023\023\023\023\
\\023\023\023\023\023\023\023\023\023\023\023\000\000\000\000\023\
\\000\023\023\023\023\023\023\023\023\023\023\023\023\023\023\023\
\\023\023\083\023\023\023\023\023\023\023\023\000\000\000\000\000\
\\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000"
),
 (83, 
"\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\\000\000\000\000\024\000\000\000\000\000\000\000\000\023\023\022\
\\023\023\023\023\023\023\023\023\023\023\000\000\000\000\000\000\
\\000\023\023\023\023\023\023\023\023\023\023\023\023\023\023\023\
\\023\023\023\023\023\023\023\023\023\023\023\000\000\000\000\023\
\\000\023\023\023\023\023\023\023\023\084\023\023\023\023\023\023\
\\023\023\023\023\023\023\023\023\023\023\023\000\000\000\000\000\
\\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000"
),
 (84, 
"\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\\000\000\000\000\024\000\000\000\000\000\000\000\000\023\023\022\
\\023\023\023\023\023\023\023\023\023\023\000\000\000\000\000\000\
\\000\023\023\023\023\023\023\023\023\023\023\023\023\023\023\023\
\\023\023\023\023\023\023\023\023\023\023\023\000\000\000\000\023\
\\000\023\023\023\023\023\023\023\023\023\023\023\023\085\023\023\
\\023\023\023\023\023\023\023\023\023\023\023\000\000\000\000\000\
\\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000"
),
 (89, 
"\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\\000\000\000\000\096\000\000\000\000\000\000\000\000\000\092\000\
\\090\090\090\090\090\090\090\090\090\090\000\000\000\000\000\000\
\\000\090\090\090\090\090\090\090\090\090\090\090\090\090\090\090\
\\090\090\090\090\090\090\090\090\090\090\090\000\000\000\000\090\
\\000\090\090\090\090\090\090\090\090\090\090\090\090\090\090\090\
\\090\090\090\090\090\090\090\090\090\090\090\000\000\000\000\000\
\\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000"
),
 (90, 
"\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\\000\000\000\000\093\000\000\000\000\000\000\000\000\090\090\091\
\\090\090\090\090\090\090\090\090\090\090\000\000\000\000\000\000\
\\000\090\090\090\090\090\090\090\090\090\090\090\090\090\090\090\
\\090\090\090\090\090\090\090\090\090\090\090\000\000\000\000\090\
\\000\090\090\090\090\090\090\090\090\090\090\090\090\090\090\090\
\\090\090\090\090\090\090\090\090\090\090\090\000\000\000\000\000\
\\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000"
),
 (93, 
"\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\\000\000\000\000\000\000\000\000\094\000\000\000\000\000\000\000\
\\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000"
),
 (94, 
"\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\\000\095\095\095\095\095\095\095\095\095\095\095\095\095\095\095\
\\095\095\095\095\095\095\095\095\095\095\095\000\000\000\000\095\
\\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000"
),
 (95, 
"\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\\000\000\000\000\000\000\000\000\000\090\000\000\000\000\000\000\
\\095\095\095\095\095\095\095\095\095\095\000\000\000\000\000\000\
\\000\095\095\095\095\095\095\095\095\095\095\095\095\095\095\095\
\\095\095\095\095\095\095\095\095\095\095\095\000\000\000\000\095\
\\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000"
),
 (96, 
"\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\\000\000\000\000\000\000\000\000\097\000\000\000\000\000\000\000\
\\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000"
),
 (97, 
"\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\\000\098\098\098\098\098\098\098\098\098\098\098\098\098\098\098\
\\098\098\098\098\098\098\098\098\098\098\098\000\000\000\000\098\
\\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000"
),
 (98, 
"\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\\000\000\000\000\000\000\000\000\000\090\000\000\000\000\000\000\
\\098\098\098\098\098\098\098\098\098\098\000\000\000\000\000\000\
\\000\098\098\098\098\098\098\098\098\098\098\098\098\098\098\098\
\\098\098\098\098\098\098\098\098\098\098\098\000\000\000\000\098\
\\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000"
),
 (101, 
"\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\\000\000\000\000\000\000\000\000\000\000\102\000\000\000\000\000\
\\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000"
),
 (102, 
"\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\\000\000\000\104\000\000\000\000\000\103\000\000\000\000\000\000\
\\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000"
),
 (104, 
"\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\\000\000\000\000\000\000\000\000\000\000\000\000\105\000\000\000\
\\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000"
),
 (105, 
"\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\\000\000\000\000\000\000\000\000\000\106\000\000\000\000\000\000\
\\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000"
),
 (106, 
"\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\\000\000\000\000\000\000\000\000\000\000\000\000\000\000\107\000\
\\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000"
),
 (107, 
"\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\\000\000\000\000\000\108\000\000\000\000\000\000\000\000\000\000\
\\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000"
),
 (108, 
"\000\000\000\000\000\000\000\000\000\109\000\109\109\000\000\000\
\\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\\109\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000"
),
 (112, 
"\000\000\000\000\000\000\000\000\000\113\000\113\113\000\000\000\
\\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\\113\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000"
),
 (114, 
"\000\000\000\000\000\000\000\000\000\000\115\000\000\000\000\000\
\\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000"
),
 (117, 
"\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\\118\118\118\118\118\118\118\118\118\118\118\118\118\118\118\118\
\\118\118\118\118\118\118\118\118\118\118\118\118\118\118\118\118\
\\118\118\118\118\118\118\118\118\118\118\118\118\118\118\118\118\
\\118\118\118\118\118\118\118\118\118\118\118\118\118\118\118\118\
\\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000"
),
 (118, 
"\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\\119\119\119\119\119\119\119\119\119\119\119\119\119\119\119\119\
\\119\119\119\119\119\119\119\119\119\119\119\119\119\119\119\119\
\\119\119\119\119\119\119\119\119\119\119\119\119\119\119\119\119\
\\119\119\119\119\119\119\119\119\119\119\119\119\119\119\119\119\
\\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000"
),
 (119, 
"\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\\120\120\120\120\120\120\120\120\120\120\120\120\120\120\120\120\
\\120\120\120\120\120\120\120\120\120\120\120\120\120\120\120\120\
\\120\120\120\120\120\120\120\120\120\120\120\120\120\120\120\120\
\\120\120\120\120\120\120\120\120\120\120\120\120\120\120\120\120\
\\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000"
),
 (121, 
"\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\\122\122\122\122\122\122\122\122\122\122\122\122\122\122\122\122\
\\122\122\122\122\122\122\122\122\122\122\122\122\122\122\122\122\
\\122\122\122\122\122\122\122\122\122\122\122\122\122\122\122\122\
\\122\122\122\122\122\122\122\122\122\122\122\122\122\122\122\122\
\\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000"
),
 (122, 
"\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\\123\123\123\123\123\123\123\123\123\123\123\123\123\123\123\123\
\\123\123\123\123\123\123\123\123\123\123\123\123\123\123\123\123\
\\123\123\123\123\123\123\123\123\123\123\123\123\123\123\123\123\
\\123\123\123\123\123\123\123\123\123\123\123\123\123\123\123\123\
\\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000"
),
 (124, 
"\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\\125\125\125\125\125\125\125\125\125\125\125\125\125\125\125\125\
\\125\125\125\125\125\125\125\125\125\125\125\125\125\125\125\125\
\\125\125\125\125\125\125\125\125\125\125\125\125\125\125\125\125\
\\125\125\125\125\125\125\125\125\125\125\125\125\125\125\125\125\
\\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000"
),
 (127, 
"\000\000\000\000\000\000\000\000\000\157\159\157\157\158\000\000\
\\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\\157\000\156\000\000\000\000\000\000\000\000\000\000\000\000\000\
\\153\153\153\153\153\153\153\153\153\153\000\000\000\000\000\000\
\\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\\000\000\000\000\000\144\000\000\000\000\000\000\143\000\140\000\
\\000\139\138\000\000\000\137\000\000\000\000\000\000\000\136\000\
\\000\000\135\000\134\129\128\000\000\000\000\000\000\000\000\000\
\\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000"
),
 (129, 
"\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\\130\130\130\130\130\130\130\130\130\130\000\000\000\000\000\000\
\\000\130\130\130\130\130\130\000\000\000\000\000\000\000\000\000\
\\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\\000\130\130\130\130\130\130\000\000\000\000\000\000\000\000\000\
\\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000"
),
 (130, 
"\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\\131\131\131\131\131\131\131\131\131\131\000\000\000\000\000\000\
\\000\131\131\131\131\131\131\000\000\000\000\000\000\000\000\000\
\\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\\000\131\131\131\131\131\131\000\000\000\000\000\000\000\000\000\
\\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000"
),
 (131, 
"\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\\132\132\132\132\132\132\132\132\132\132\000\000\000\000\000\000\
\\000\132\132\132\132\132\132\000\000\000\000\000\000\000\000\000\
\\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\\000\132\132\132\132\132\132\000\000\000\000\000\000\000\000\000\
\\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000"
),
 (132, 
"\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\\133\133\133\133\133\133\133\133\133\133\000\000\000\000\000\000\
\\000\133\133\133\133\133\133\000\000\000\000\000\000\000\000\000\
\\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\\000\133\133\133\133\133\133\000\000\000\000\000\000\000\000\000\
\\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000"
),
 (140, 
"\141\141\141\141\141\141\141\141\141\141\000\141\141\141\141\141\
\\141\141\141\141\141\141\141\141\141\141\141\141\141\141\141\141\
\\141\141\141\141\141\141\141\141\141\141\141\141\141\141\141\141\
\\141\141\141\141\141\141\141\141\141\141\141\141\141\141\141\141\
\\142\142\142\142\142\142\142\142\142\142\142\142\142\142\142\142\
\\142\142\142\142\142\142\142\142\142\142\142\142\142\142\142\142\
\\141\141\141\141\141\141\141\141\141\141\141\141\141\141\141\141\
\\141\141\141\141\141\141\141\141\141\141\141\141\141\141\141\141\
\\141\141\141\141\141\141\141\141\141\141\141\141\141\141\141\141\
\\141\141\141\141\141\141\141\141\141\141\141\141\141\141\141\141\
\\141\141\141\141\141\141\141\141\141\141\141\141\141\141\141\141\
\\141\141\141\141\141\141\141\141\141\141\141\141\141\141\141\141\
\\141\141\141\141\141\141\141\141\141\141\141\141\141\141\141\141\
\\141\141\141\141\141\141\141\141\141\141\141\141\141\141\141\141\
\\141\141\141\141\141\141\141\141\141\141\141\141\141\141\141\141\
\\141\141\141\141\141\141\141\141\141\141\141\141\141\141\141\141"
),
 (144, 
"\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\\145\145\145\145\145\145\145\145\145\145\000\000\000\000\000\000\
\\000\145\145\145\145\145\145\000\000\000\000\000\000\000\000\000\
\\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\\000\145\145\145\145\145\145\000\000\000\000\000\000\000\000\000\
\\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000"
),
 (145, 
"\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\\146\146\146\146\146\146\146\146\146\146\000\000\000\000\000\000\
\\000\146\146\146\146\146\146\000\000\000\000\000\000\000\000\000\
\\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\\000\146\146\146\146\146\146\000\000\000\000\000\000\000\000\000\
\\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000"
),
 (146, 
"\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\\147\147\147\147\147\147\147\147\147\147\000\000\000\000\000\000\
\\000\147\147\147\147\147\147\000\000\000\000\000\000\000\000\000\
\\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\\000\147\147\147\147\147\147\000\000\000\000\000\000\000\000\000\
\\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000"
),
 (147, 
"\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\\148\148\148\148\148\148\148\148\148\148\000\000\000\000\000\000\
\\000\148\148\148\148\148\148\000\000\000\000\000\000\000\000\000\
\\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\\000\148\148\148\148\148\148\000\000\000\000\000\000\000\000\000\
\\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000"
),
 (148, 
"\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\\149\149\149\149\149\149\149\149\149\149\000\000\000\000\000\000\
\\000\149\149\149\149\149\149\000\000\000\000\000\000\000\000\000\
\\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\\000\149\149\149\149\149\149\000\000\000\000\000\000\000\000\000\
\\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000"
),
 (149, 
"\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\\150\150\150\150\150\150\150\150\150\150\000\000\000\000\000\000\
\\000\150\150\150\150\150\150\000\000\000\000\000\000\000\000\000\
\\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\\000\150\150\150\150\150\150\000\000\000\000\000\000\000\000\000\
\\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000"
),
 (150, 
"\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\\151\151\151\151\151\151\151\151\151\151\000\000\000\000\000\000\
\\000\151\151\151\151\151\151\000\000\000\000\000\000\000\000\000\
\\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\\000\151\151\151\151\151\151\000\000\000\000\000\000\000\000\000\
\\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000"
),
 (151, 
"\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\\152\152\152\152\152\152\152\152\152\152\000\000\000\000\000\000\
\\000\152\152\152\152\152\152\000\000\000\000\000\000\000\000\000\
\\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\\000\152\152\152\152\152\152\000\000\000\000\000\000\000\000\000\
\\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000"
),
 (153, 
"\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\\154\154\154\154\154\154\154\154\154\154\000\000\000\000\000\000\
\\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000"
),
 (154, 
"\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\\155\155\155\155\155\155\155\155\155\155\000\000\000\000\000\000\
\\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000"
),
 (157, 
"\000\000\000\000\000\000\000\000\000\157\000\157\157\000\000\000\
\\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\\157\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000"
),
 (158, 
"\000\000\000\000\000\000\000\000\000\000\159\000\000\000\000\000\
\\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000"
),
 (161, 
"\000\000\000\000\000\000\000\000\000\000\162\000\000\000\000\000\
\\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000"
),
 (165, 
"\000\000\000\000\000\000\000\000\000\166\000\166\166\000\000\000\
\\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\\166\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000"
),
 (167, 
"\000\000\000\000\000\000\000\000\000\000\168\000\000\000\000\000\
\\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000"
),
 (170, 
"\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\\000\000\000\000\000\000\000\000\000\171\000\000\000\000\000\000\
\\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000"
),
 (172, 
"\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\\000\000\000\000\000\000\000\000\000\000\173\000\000\000\000\000\
\\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000"
),
 (173, 
"\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\\000\000\000\000\000\000\000\000\000\174\000\000\000\000\000\000\
\\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000"
),
 (175, 
"\000\000\000\000\000\000\000\000\000\000\176\000\000\000\000\000\
\\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000"
),
 (178, 
"\000\000\000\000\000\000\000\000\000\000\179\000\000\000\000\000\
\\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000"
),
 (181, 
"\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\\000\000\000\000\000\000\000\000\000\000\000\000\000\000\183\000\
\\182\182\182\182\182\182\182\182\182\182\000\000\000\000\000\000\
\\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000"
),
 (183, 
"\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\\184\184\184\184\184\184\184\184\184\184\000\000\000\000\000\000\
\\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000"
),
 (185, 
"\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\\000\000\000\000\000\000\000\000\000\186\000\000\000\000\000\000\
\\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000"
),
 (187, 
"\000\000\000\000\000\000\000\000\000\190\000\190\190\000\000\000\
\\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\\190\000\189\000\000\000\000\000\000\000\188\000\000\000\000\000\
\\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000"
),
 (191, 
"\192\192\192\192\192\192\192\192\192\192\192\192\192\192\192\192\
\\192\192\192\192\192\192\192\192\192\192\192\192\192\192\192\192\
\\192\192\193\192\192\192\192\192\192\192\192\192\192\192\192\192\
\\192\192\192\192\192\192\192\192\192\192\192\192\192\192\192\192\
\\192\192\192\192\192\192\192\192\192\192\192\192\192\192\192\192\
\\192\192\192\192\192\192\192\192\192\192\192\192\192\192\192\192\
\\192\192\192\192\192\192\192\192\192\192\192\192\192\192\192\192\
\\192\192\192\192\192\192\192\192\192\192\192\192\192\192\192\192\
\\192\192\192\192\192\192\192\192\192\192\192\192\192\192\192\192\
\\192\192\192\192\192\192\192\192\192\192\192\192\192\192\192\192\
\\192\192\192\192\192\192\192\192\192\192\192\192\192\192\192\192\
\\192\192\192\192\192\192\192\192\192\192\192\192\192\192\192\192\
\\192\192\192\192\192\192\192\192\192\192\192\192\192\192\192\192\
\\192\192\192\192\192\192\192\192\192\192\192\192\192\192\192\192\
\\192\192\192\192\192\192\192\192\192\192\192\192\192\192\192\192\
\\192\192\192\192\192\192\192\192\192\192\192\192\192\192\192\192"
),
 (195, 
"\000\000\000\000\000\000\000\000\000\196\000\196\196\000\000\000\
\\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\\196\000\000\000\000\000\000\000\000\000\188\000\000\000\000\000\
\\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000"
),
(0, "")]
fun f x = x 
val s = Pervasive.List.map f (Pervasive.List.rev (tl (Pervasive.List.rev s))) 
exception LexHackingError 
fun look ((j,x)::r, i: int) = if i = j then x else look(r, i) 
  | look ([], i) = raise LexHackingError
fun g {fin=x, trans=i} = {fin=x, trans=look(s,i)} 
in Vector.fromList(Pervasive.List.map g 
[{fin = [], trans = 0},
{fin = [], trans = 1},
{fin = [], trans = 1},
{fin = [], trans = 3},
{fin = [], trans = 3},
{fin = [], trans = 5},
{fin = [], trans = 5},
{fin = [], trans = 7},
{fin = [], trans = 7},
{fin = [], trans = 9},
{fin = [], trans = 9},
{fin = [], trans = 11},
{fin = [], trans = 11},
{fin = [], trans = 13},
{fin = [], trans = 13},
{fin = [], trans = 15},
{fin = [], trans = 15},
{fin = [], trans = 17},
{fin = [], trans = 17},
{fin = [(N 354)], trans = 0},
{fin = [(N 96),(N 152),(N 354)], trans = 20},
{fin = [(N 96),(N 152)], trans = 20},
{fin = [], trans = 22},
{fin = [(N 152)], trans = 23},
{fin = [], trans = 24},
{fin = [], trans = 25},
{fin = [], trans = 26},
{fin = [(N 152)], trans = 23},
{fin = [], trans = 28},
{fin = [], trans = 29},
{fin = [], trans = 30},
{fin = [(N 96)], trans = 31},
{fin = [(N 96),(N 152),(N 354)], trans = 32},
{fin = [(N 96),(N 152)], trans = 33},
{fin = [(N 96),(N 152)], trans = 34},
{fin = [(N 96),(N 152)], trans = 35},
{fin = [(N 96),(N 152)], trans = 36},
{fin = [(N 96),(N 152)], trans = 37},
{fin = [(N 96),(N 152)], trans = 38},
{fin = [(N 96),(N 152)], trans = 39},
{fin = [(N 93),(N 96),(N 152)], trans = 20},
{fin = [(N 96),(N 152)], trans = 41},
{fin = [(N 96),(N 152)], trans = 42},
{fin = [(N 96),(N 152)], trans = 43},
{fin = [(N 96),(N 152)], trans = 44},
{fin = [(N 96),(N 152)], trans = 45},
{fin = [(N 96),(N 152)], trans = 46},
{fin = [(N 96),(N 152)], trans = 47},
{fin = [(N 83),(N 96),(N 152)], trans = 20},
{fin = [(N 96),(N 152),(N 354)], trans = 49},
{fin = [(N 96),(N 152)], trans = 50},
{fin = [(N 96),(N 152)], trans = 51},
{fin = [(N 73),(N 96),(N 152)], trans = 20},
{fin = [(N 96),(N 152),(N 354)], trans = 53},
{fin = [(N 96),(N 152)], trans = 54},
{fin = [(N 96),(N 152)], trans = 55},
{fin = [(N 96),(N 152)], trans = 56},
{fin = [(N 68),(N 96),(N 152)], trans = 20},
{fin = [(N 96),(N 152)], trans = 58},
{fin = [(N 62),(N 96),(N 152)], trans = 20},
{fin = [(N 96),(N 152),(N 354)], trans = 60},
{fin = [(N 58),(N 96),(N 152)], trans = 20},
{fin = [(N 96),(N 152),(N 354)], trans = 62},
{fin = [(N 96),(N 152)], trans = 63},
{fin = [(N 96),(N 152)], trans = 64},
{fin = [(N 96),(N 152)], trans = 65},
{fin = [(N 96),(N 152)], trans = 66},
{fin = [(N 96),(N 152)], trans = 67},
{fin = [(N 55),(N 96),(N 152)], trans = 20},
{fin = [(N 96),(N 152),(N 354)], trans = 69},
{fin = [(N 96),(N 152)], trans = 70},
{fin = [(N 47),(N 96),(N 152)], trans = 20},
{fin = [(N 96),(N 152),(N 354)], trans = 72},
{fin = [(N 96),(N 152)], trans = 73},
{fin = [(N 37),(N 96),(N 152)], trans = 74},
{fin = [(N 96),(N 152)], trans = 75},
{fin = [(N 43),(N 96),(N 152)], trans = 20},
{fin = [(N 96),(N 152),(N 354)], trans = 77},
{fin = [(N 96),(N 152)], trans = 78},
{fin = [(N 33),(N 96),(N 152)], trans = 20},
{fin = [(N 29),(N 96),(N 152)], trans = 20},
{fin = [(N 152),(N 354)], trans = 81},
{fin = [(N 152)], trans = 82},
{fin = [(N 152)], trans = 83},
{fin = [(N 152)], trans = 84},
{fin = [(N 19),(N 152)], trans = 23},
{fin = [(N 25),(N 354)], trans = 0},
{fin = [(N 23),(N 354)], trans = 0},
{fin = [(N 152),(N 354)], trans = 23},
{fin = [(N 354)], trans = 89},
{fin = [(N 152)], trans = 90},
{fin = [], trans = 89},
{fin = [(N 152)], trans = 90},
{fin = [], trans = 93},
{fin = [], trans = 94},
{fin = [], trans = 95},
{fin = [], trans = 96},
{fin = [], trans = 97},
{fin = [], trans = 98},
{fin = [(N 152),(N 354)], trans = 23},
{fin = [(N 21),(N 354)], trans = 0},
{fin = [(N 354)], trans = 101},
{fin = [(N 281)], trans = 102},
{fin = [(N 278)], trans = 0},
{fin = [], trans = 104},
{fin = [], trans = 105},
{fin = [], trans = 106},
{fin = [], trans = 107},
{fin = [], trans = 108},
{fin = [(N 321)], trans = 108},
{fin = [(N 354)], trans = 28},
{fin = [(N 154),(N 354)], trans = 0},
{fin = [(N 8),(N 354)], trans = 112},
{fin = [(N 8)], trans = 112},
{fin = [(N 13),(N 354)], trans = 114},
{fin = [(N 13)], trans = 0},
{fin = [(N 256)], trans = 0},
{fin = [(N 256)], trans = 117},
{fin = [], trans = 118},
{fin = [], trans = 119},
{fin = [(N 173)], trans = 0},
{fin = [(N 256)], trans = 121},
{fin = [], trans = 122},
{fin = [(N 168)], trans = 0},
{fin = [(N 256)], trans = 124},
{fin = [(N 164)], trans = 0},
{fin = [(N 161),(N 256)], trans = 0},
{fin = [(N 249),(N 256)], trans = 127},
{fin = [(N 188)], trans = 0},
{fin = [], trans = 129},
{fin = [], trans = 130},
{fin = [], trans = 131},
{fin = [], trans = 132},
{fin = [(N 214)], trans = 0},
{fin = [(N 182)], trans = 0},
{fin = [(N 194)], trans = 0},
{fin = [(N 185)], trans = 0},
{fin = [(N 191)], trans = 0},
{fin = [(N 179)], trans = 0},
{fin = [(N 176)], trans = 0},
{fin = [], trans = 140},
{fin = [(N 202)], trans = 0},
{fin = [(N 198),(N 202)], trans = 0},
{fin = [(N 231)], trans = 0},
{fin = [], trans = 144},
{fin = [], trans = 145},
{fin = [], trans = 146},
{fin = [], trans = 147},
{fin = [], trans = 148},
{fin = [], trans = 149},
{fin = [], trans = 150},
{fin = [], trans = 151},
{fin = [(N 225)], trans = 0},
{fin = [], trans = 153},
{fin = [], trans = 154},
{fin = [(N 207)], trans = 0},
{fin = [(N 228)], trans = 0},
{fin = [(N 241)], trans = 157},
{fin = [(N 247)], trans = 158},
{fin = [(N 247)], trans = 0},
{fin = [(N 156),(N 256)], trans = 0},
{fin = [(N 254),(N 256)], trans = 161},
{fin = [(N 254)], trans = 0},
{fin = [(N 274)], trans = 0},
{fin = [(N 272),(N 274)], trans = 0},
{fin = [(N 265),(N 274)], trans = 165},
{fin = [(N 265)], trans = 165},
{fin = [(N 270),(N 274)], trans = 167},
{fin = [(N 270)], trans = 0},
{fin = [(N 305)], trans = 0},
{fin = [(N 305)], trans = 170},
{fin = [(N 298)], trans = 0},
{fin = [(N 305)], trans = 172},
{fin = [(N 295)], trans = 173},
{fin = [(N 292)], trans = 0},
{fin = [(N 303),(N 305)], trans = 175},
{fin = [(N 303)], trans = 0},
{fin = [(N 288)], trans = 0},
{fin = [(N 286),(N 288)], trans = 178},
{fin = [(N 286)], trans = 0},
{fin = [(N 352)], trans = 0},
{fin = [(N 330),(N 352)], trans = 181},
{fin = [(N 330)], trans = 181},
{fin = [], trans = 183},
{fin = [(N 327)], trans = 183},
{fin = [(N 352)], trans = 185},
{fin = [(N 350)], trans = 0},
{fin = [(N 352)], trans = 187},
{fin = [], trans = 185},
{fin = [(N 340)], trans = 0},
{fin = [], trans = 187},
{fin = [(N 352)], trans = 191},
{fin = [], trans = 191},
{fin = [(N 343)], trans = 0},
{fin = [(N 343),(N 352)], trans = 0},
{fin = [(N 352)], trans = 195},
{fin = [], trans = 195}])
end
structure StartStates =
	struct
	datatype yystartstate = STARTSTATE of int

(* start state definitions *)

val BLOCK_COMMENT = STARTSTATE 7;
val INITIAL = STARTSTATE 1;
val LINE_COMMENT = STARTSTATE 9;
val LINE_DIR1 = STARTSTATE 11;
val LINE_DIR2 = STARTSTATE 13;
val LINE_DIR3 = STARTSTATE 15;
val LINE_DIR4 = STARTSTATE 17;
val TEXT = STARTSTATE 3;
val TEXT_FMT = STARTSTATE 5;

end
type result = UserDeclarations.lexresult
	exception LexerError (* raised if illegal leaf action tried *)
end

structure YYPosInt : INTEGER = Int
fun makeLexer yyinput =
let	val yygone0= YYPosInt.fromInt ~1
	val yyb = ref "\n" 		(* buffer *)
	val yybl = ref 1		(*buffer length *)
	val yybufpos = ref 1		(* location of next character to use *)
	val yygone = ref yygone0	(* position in file of beginning of buffer *)
	val yydone = ref false		(* eof found yet? *)
	val yybegin = ref 1		(*Current 'start state' for lexer *)

	val YYBEGIN = fn (Internal.StartStates.STARTSTATE x) =>
		 yybegin := x

fun lex (yyarg as ((*#line 179.7 "mlb.lex"*){source}(*#line 2535.1 "mlb.lex.sml"*)
)) =
let fun continue() : Internal.result = 
  let fun scan (s,AcceptingLeaves : Internal.yyfinstate list list,l,i0) =
	let fun action (i,nil) = raise LexError
	| action (i,nil::l) = action (i-1,l)
	| action (i,(node::acts)::l) =
		case node of
		    Internal.N yyk => 
			(let fun yymktext() = String.substring(!yyb,i0,i-i0)
			     val yypos = YYPosInt.+(YYPosInt.fromInt i0, !yygone)
			open UserDeclarations Internal.StartStates
 in (yybufpos := i; case yyk of 

			(* Application actions *)

  13 => let val yytext=yymktext() in (*#line 203.21 "mlb.lex"*)Source.newline (source, lastPos (yypos, yytext)); continue ()(*#line 2551.1 "mlb.lex.sml"*)
 end
| 152 => let val yytext=yymktext() in (*#line 225.21 "mlb.lex"*)tok' (Tokens.FILE, yytext, source, yypos)(*#line 2553.1 "mlb.lex.sml"*)
 end
| 154 => ((*#line 228.5 "mlb.lex"*)startText (Source.getPos (source, yypos), fn (s, l, r) =>
               (YYBEGIN INITIAL;
                Tokens.STRING (s, l, r)))
    ; YYBEGIN TEXT
    ; continue ()(*#line 2559.1 "mlb.lex.sml"*)
)
| 156 => let val yytext=yymktext() in (*#line 234.22 "mlb.lex"*)finishText (Source.getPos (source, lastPos (yypos, yytext)))(*#line 2561.1 "mlb.lex.sml"*)
 end
| 161 => let val yytext=yymktext() in (*#line 236.22 "mlb.lex"*)addTextString yytext; continue ()(*#line 2563.1 "mlb.lex.sml"*)
 end
| 164 => let val yytext=yymktext() in (*#line 238.22 "mlb.lex"*)addTextUTF8 (source, yypos, yytext); continue()(*#line 2565.1 "mlb.lex.sml"*)
 end
| 168 => let val yytext=yymktext() in (*#line 240.22 "mlb.lex"*)addTextUTF8 (source, yypos, yytext); continue()(*#line 2567.1 "mlb.lex.sml"*)
 end
| 173 => let val yytext=yymktext() in (*#line 242.22 "mlb.lex"*)addTextUTF8 (source, yypos, yytext); continue()(*#line 2569.1 "mlb.lex.sml"*)
 end
| 176 => ((*#line 243.22 "mlb.lex"*)addTextChar #"\a"; continue ()(*#line 2571.1 "mlb.lex.sml"*)
)
| 179 => ((*#line 244.22 "mlb.lex"*)addTextChar #"\b"; continue ()(*#line 2573.1 "mlb.lex.sml"*)
)
| 182 => ((*#line 245.22 "mlb.lex"*)addTextChar #"\t"; continue ()(*#line 2575.1 "mlb.lex.sml"*)
)
| 185 => ((*#line 246.22 "mlb.lex"*)addTextChar #"\n"; continue ()(*#line 2577.1 "mlb.lex.sml"*)
)
| 188 => ((*#line 247.22 "mlb.lex"*)addTextChar #"\v"; continue ()(*#line 2579.1 "mlb.lex.sml"*)
)
| 19 => let val yytext=yymktext() in (*#line 205.22 "mlb.lex"*)tok (Tokens.PRIM, yytext, source, yypos)(*#line 2581.1 "mlb.lex.sml"*)
 end
| 191 => ((*#line 248.22 "mlb.lex"*)addTextChar #"\f"; continue ()(*#line 2583.1 "mlb.lex.sml"*)
)
| 194 => ((*#line 249.22 "mlb.lex"*)addTextChar #"\r"; continue ()(*#line 2585.1 "mlb.lex.sml"*)
)
| 198 => let val yytext=yymktext() in (*#line 250.22 "mlb.lex"*)addTextChar (Char.chr(Char.ord(String.sub(yytext, 2)) - Char.ord #"@"));
                     continue ()(*#line 2588.1 "mlb.lex.sml"*)
 end
| 202 => ((*#line 252.22 "mlb.lex"*)error (source, yypos, yypos + 2, "Illegal control escape in text constant; must be one of @ABCDEFGHIJKLMNOPQRSTUVWXYZ[\\]^_");
                     continue ()(*#line 2591.1 "mlb.lex.sml"*)
)
| 207 => let val yytext=yymktext() in (*#line 254.22 "mlb.lex"*)addTextNumEsc (source, yypos, yytext, 1,
                                    StringCvt.DEC)
                     ; continue ()(*#line 2595.1 "mlb.lex.sml"*)
 end
| 21 => let val yytext=yymktext() in (*#line 207.18 "mlb.lex"*)tok (Tokens.COMMA, yytext, source, yypos)(*#line 2597.1 "mlb.lex.sml"*)
 end
| 214 => let val yytext=yymktext() in (*#line 258.22 "mlb.lex"*)addTextNumEsc (source, yypos, yytext, 2,
                                    StringCvt.HEX)
                     ; continue ()(*#line 2601.1 "mlb.lex.sml"*)
 end
| 225 => let val yytext=yymktext() in (*#line 262.22 "mlb.lex"*)addTextNumEsc (source, yypos, yytext, 2,
                                    StringCvt.HEX)
                     ; continue ()(*#line 2605.1 "mlb.lex.sml"*)
 end
| 228 => ((*#line 265.22 "mlb.lex"*)addTextString "\""; continue ()(*#line 2607.1 "mlb.lex.sml"*)
)
| 23 => let val yytext=yymktext() in (*#line 208.18 "mlb.lex"*)tok (Tokens.SEMICOLON, yytext, source, yypos)(*#line 2609.1 "mlb.lex.sml"*)
 end
| 231 => ((*#line 266.22 "mlb.lex"*)addTextString "\\"; continue ()(*#line 2611.1 "mlb.lex.sml"*)
)
| 241 => ((*#line 267.22 "mlb.lex"*)YYBEGIN TEXT_FMT; continue ()(*#line 2613.1 "mlb.lex.sml"*)
)
| 247 => let val yytext=yymktext() in (*#line 268.22 "mlb.lex"*)Source.newline (source, lastPos (yypos, yytext)); YYBEGIN TEXT_FMT; continue ()(*#line 2615.1 "mlb.lex.sml"*)
 end
| 249 => ((*#line 269.22 "mlb.lex"*)error (source, yypos, yypos + 1, "Illegal escape in text constant")
                     ; continue ()(*#line 2618.1 "mlb.lex.sml"*)
)
| 25 => let val yytext=yymktext() in (*#line 209.18 "mlb.lex"*)tok (Tokens.EQUALOP, yytext, source, yypos)(*#line 2620.1 "mlb.lex.sml"*)
 end
| 254 => let val yytext=yymktext() in (*#line 271.22 "mlb.lex"*)error (source, yypos, lastPos (yypos, yytext), "Unclosed text constant at end of line")
                     ; Source.newline (source, lastPos (yypos, yytext))
                     ; continue ()(*#line 2624.1 "mlb.lex.sml"*)
 end
| 256 => ((*#line 274.22 "mlb.lex"*)error (source, yypos, yypos, "Illegal character in text constant")
                     ; continue ()(*#line 2627.1 "mlb.lex.sml"*)
)
| 265 => ((*#line 277.22 "mlb.lex"*)continue ()(*#line 2629.1 "mlb.lex.sml"*)
)
| 270 => let val yytext=yymktext() in (*#line 278.22 "mlb.lex"*)Source.newline (source, lastPos (yypos, yytext)); continue ()(*#line 2631.1 "mlb.lex.sml"*)
 end
| 272 => ((*#line 279.22 "mlb.lex"*)YYBEGIN TEXT; continue ()(*#line 2633.1 "mlb.lex.sml"*)
)
| 274 => ((*#line 280.22 "mlb.lex"*)error (source, yypos, yypos, "Illegal formatting character in text continuation")
                     ; continue ()(*#line 2636.1 "mlb.lex.sml"*)
)
| 278 => ((*#line 285.5 "mlb.lex"*)startComment (source, yypos, fn () =>
                  YYBEGIN INITIAL)
    ; YYBEGIN LINE_COMMENT
    ; continue ()(*#line 2641.1 "mlb.lex.sml"*)
)
| 281 => ((*#line 290.5 "mlb.lex"*)startComment (source, yypos, fn () =>
                  YYBEGIN INITIAL)
    ; YYBEGIN BLOCK_COMMENT
    ; continue ()(*#line 2646.1 "mlb.lex.sml"*)
)
| 286 => let val yytext=yymktext() in (*#line 296.5 "mlb.lex"*)finishComment (lastPos (yypos, yytext))
    ; Source.newline (source, lastPos (yypos, yytext))
    ; continue ()(*#line 2650.1 "mlb.lex.sml"*)
 end
| 288 => ((*#line 300.5 "mlb.lex"*)continue ()(*#line 2652.1 "mlb.lex.sml"*)
)
| 29 => let val yytext=yymktext() in (*#line 211.20 "mlb.lex"*)tok (Tokens.AND, yytext, source, yypos)(*#line 2654.1 "mlb.lex.sml"*)
 end
| 292 => ((*#line 303.5 "mlb.lex"*)startComment (source, yypos, fn () =>
                  YYBEGIN BLOCK_COMMENT)
    ; YYBEGIN LINE_COMMENT
    ; continue ()(*#line 2659.1 "mlb.lex.sml"*)
)
| 295 => ((*#line 308.5 "mlb.lex"*)startComment (source, yypos, fn () =>
                  YYBEGIN BLOCK_COMMENT)
    ; YYBEGIN BLOCK_COMMENT
    ; continue ()(*#line 2664.1 "mlb.lex.sml"*)
)
| 298 => let val yytext=yymktext() in (*#line 313.5 "mlb.lex"*)finishComment (lastPos (yypos, yytext))
    ; continue ()(*#line 2667.1 "mlb.lex.sml"*)
 end
| 303 => let val yytext=yymktext() in (*#line 316.5 "mlb.lex"*)Source.newline (source, lastPos (yypos, yytext))
    ; continue ()(*#line 2670.1 "mlb.lex.sml"*)
 end
| 305 => ((*#line 319.5 "mlb.lex"*)continue ()(*#line 2672.1 "mlb.lex.sml"*)
)
| 321 => ((*#line 323.5 "mlb.lex"*)startLineDir (source, yypos, fn () =>
                  YYBEGIN INITIAL)
    ; YYBEGIN LINE_DIR1
    ; continue ()(*#line 2677.1 "mlb.lex.sml"*)
)
| 327 => let val yytext=yymktext() in (*#line 329.5 "mlb.lex"*)let
       fun err () =
          (addCommentError "Illegal line directive"
           ; YYBEGIN BLOCK_COMMENT)
     in
       case String.split (yytext, #".") of
          [line, col] => (YYBEGIN LINE_DIR2
                          ; addLineDirLineCol (valOf (Int.fromString line), valOf (Int.fromString col))
                            handle Overflow => err () | Option => err ()
                          ; continue ())
        | _ => (err (); continue ())
     end(*#line 2690.1 "mlb.lex.sml"*)
 end
| 33 => let val yytext=yymktext() in (*#line 212.20 "mlb.lex"*)tok (Tokens.ANN, yytext, source, yypos)(*#line 2692.1 "mlb.lex.sml"*)
 end
| 330 => let val yytext=yymktext() in (*#line 342.5 "mlb.lex"*)let
       fun err () =
          (addCommentError "Illegal line directive"
           ; YYBEGIN BLOCK_COMMENT)
       val line = yytext
       val col = 1
     in
       YYBEGIN LINE_DIR2
       ; addLineDirLineCol (valOf (Int.fromString line), col)
         handle Overflow => err () | Option => err ()
       ; continue ()
     end(*#line 2705.1 "mlb.lex.sml"*)
 end
| 340 => ((*#line 355.5 "mlb.lex"*)YYBEGIN LINE_DIR3
    ; continue ()(*#line 2708.1 "mlb.lex.sml"*)
)
| 343 => let val yytext=yymktext() in (*#line 358.5 "mlb.lex"*)addLineDirFile (String.dropLast yytext)
    ; YYBEGIN LINE_DIR4
    ; continue ()(*#line 2712.1 "mlb.lex.sml"*)
 end
| 350 => let val yytext=yymktext() in (*#line 362.5 "mlb.lex"*)finishLineDir (source, lastPos (yypos, yytext))
    ; continue ()(*#line 2715.1 "mlb.lex.sml"*)
 end
| 352 => ((*#line 365.5 "mlb.lex"*)addCommentError "Illegal line directive"
    ; YYBEGIN BLOCK_COMMENT
    ; continue ()(*#line 2719.1 "mlb.lex.sml"*)
)
| 354 => ((*#line 371.5 "mlb.lex"*)error (source, yypos, yypos, "Illegal character")
    ; continue ()(*#line 2722.1 "mlb.lex.sml"*)
)
| 37 => let val yytext=yymktext() in (*#line 213.20 "mlb.lex"*)tok (Tokens.BAS, yytext, source, yypos)(*#line 2724.1 "mlb.lex.sml"*)
 end
| 43 => let val yytext=yymktext() in (*#line 214.22 "mlb.lex"*)tok (Tokens.BASIS, yytext, source, yypos)(*#line 2726.1 "mlb.lex.sml"*)
 end
| 47 => let val yytext=yymktext() in (*#line 215.20 "mlb.lex"*)tok (Tokens.END, yytext, source, yypos)(*#line 2728.1 "mlb.lex.sml"*)
 end
| 55 => let val yytext=yymktext() in (*#line 216.24 "mlb.lex"*)tok (Tokens.FUNCTOR, yytext, source, yypos)(*#line 2730.1 "mlb.lex.sml"*)
 end
| 58 => let val yytext=yymktext() in (*#line 217.19 "mlb.lex"*)tok (Tokens.IN, yytext, source, yypos)(*#line 2732.1 "mlb.lex.sml"*)
 end
| 62 => let val yytext=yymktext() in (*#line 218.20 "mlb.lex"*)tok (Tokens.LET, yytext, source, yypos)(*#line 2734.1 "mlb.lex.sml"*)
 end
| 68 => let val yytext=yymktext() in (*#line 219.22 "mlb.lex"*)tok (Tokens.LOCAL, yytext, source, yypos)(*#line 2736.1 "mlb.lex.sml"*)
 end
| 73 => let val yytext=yymktext() in (*#line 220.21 "mlb.lex"*)tok (Tokens.OPEN, yytext, source, yypos)(*#line 2738.1 "mlb.lex.sml"*)
 end
| 8 => ((*#line 202.21 "mlb.lex"*)continue ()(*#line 2740.1 "mlb.lex.sml"*)
)
| 83 => let val yytext=yymktext() in (*#line 221.26 "mlb.lex"*)tok (Tokens.SIGNATURE, yytext, source, yypos)(*#line 2742.1 "mlb.lex.sml"*)
 end
| 93 => let val yytext=yymktext() in (*#line 222.26 "mlb.lex"*)tok (Tokens.STRUCTURE, yytext, source, yypos)(*#line 2744.1 "mlb.lex.sml"*)
 end
| 96 => let val yytext=yymktext() in (*#line 224.19 "mlb.lex"*)tok' (Tokens.ID, yytext, source, yypos)(*#line 2746.1 "mlb.lex.sml"*)
 end
| _ => raise Internal.LexerError

		) end )

	val {fin,trans} = Vector.sub(Internal.tab, s)
	val NewAcceptingLeaves = fin::AcceptingLeaves
	in if l = !yybl then
	     if trans = #trans(Vector.sub(Internal.tab,0))
	       then action(l,NewAcceptingLeaves
) else	    let val newchars= if !yydone then "" else yyinput 1024
	    in if (String.size newchars)=0
		  then (yydone := true;
		        if (l=i0) then UserDeclarations.eof yyarg
		                  else action(l,NewAcceptingLeaves))
		  else (if i0=l then yyb := newchars
		     else yyb := String.substring(!yyb,i0,l-i0)^newchars;
		     yygone := YYPosInt.+(!yygone, YYPosInt.fromInt i0);
		     yybl := String.size (!yyb);
		     scan (s,AcceptingLeaves,l-i0,0))
	    end
	  else let val NewChar = Char.ord(CharVector.sub(!yyb,l))
		val NewState = Char.ord(CharVector.sub(trans,NewChar))
		in if NewState=0 then action(l,NewAcceptingLeaves)
		else scan(NewState,NewAcceptingLeaves,l+1,i0)
	end
	end
(*
	val start= if String.substring(!yyb,!yybufpos-1,1)="\n"
then !yybegin+1 else !yybegin
*)
	in scan(!yybegin (* start *),nil,!yybufpos,!yybufpos)
    end
in continue end
  in lex
  end
end
