package latexDraw.ui.dialog;

import java.awt.Dimension;
import java.awt.FlowLayout;
import java.awt.GridLayout;
import java.awt.Toolkit;
import java.awt.event.ActionEvent;
import java.awt.event.ActionListener;

import javax.swing.*;
import javax.swing.border.CompoundBorder;
import javax.swing.border.EmptyBorder;
import javax.swing.border.TitledBorder;

import latexDraw.lang.LaTeXDrawLang;
import latexDraw.psTricks.PSTricksConstants;
import latexDraw.ui.CodePanel;
import latexDraw.ui.DrawContainer;
import latexDraw.ui.LaTeXDrawFrame;
import latexDraw.ui.components.LaTeXDrawComboBox;
import latexDraw.ui.components.XScale;
import latexDraw.util.LaTeXDrawResources;


/** 
 * This class defines the frame which allows to set the parameters of the drawing (not the figure.draw)<br>
 *<br>
 * This file is part of LaTeXDraw<br>
 * Copyright (c) 2005-2008 Arnaud BLOUIN<br>
 *<br>
 *  LaTeXDraw is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  any later version.<br>
 *<br>
 *  LaTeXDraw is distributed without any warranty; without even the 
 *  implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR 
 *  PURPOSE. See the GNU General Public License for more details.<br>
 *<br>
 * 01/20/06<br>
 * @author Arnaud BLOUIN<br>
 * @version 2.0.0<br>
 */
public class DrawPropertiesFrame extends JFrame implements ActionListener
{
	private static final long serialVersionUID = 1L;

    /** Allows to choice the unit of length (for the field pixel/unit) */
    private LaTeXDrawComboBox unitChoice;   
	
	/** The field which allows to change the number of pixels per 
	 * unit of length */
	private JSpinner pixelsPerUnitField;	
	
	/** The main frame of the program */
	private LaTeXDrawFrame parentFrame;
	
	/** The former unit selected */
	private String formerUnitChoice;
	
	/** Allows to set the caption of the drawing */
	private JTextField captionField;
	
	/** Allows to set the label of the drawing */
	private JTextField labelField;
	
	/** Allows to place the figure in a page containing only floats */
	private JRadioButton onlyWithFloatsCB;
	
	/** Allows to place the drawing at the top of the page */
	private JRadioButton topCB;
	
	/** Allows to put the drawing at the bottom of the page */
	private JRadioButton bottomCB;
	
	/** Allows to put the drawing following the code position */
	private JRadioButton hereCB;
	
	/** Obligates to put the drawing following the code position */
	private JRadioButton HERECB;
	
	/** Allows to put the drawing at the default position */
	private JRadioButton defaultPositionCB;
	
	/** Obligates to put the drawing following the code position */
	private JRadioButton centerHoriCB;
	
	/** Allows to put the drawing at the default position */
	private JRadioButton defaultPositionHoriCB;
	
	/** The list of the packages of the current drawing. */
	private JTextArea packagesDrawing;
	
	/** The list of the packages contained in the preferences. */
	private JTextArea packagesList;
	
	/** The label of the label field */
	public static final String LABEL_LABEL = LaTeXDrawLang.getDialogFrameString("DrawPropertiesFrame.0"); //$NON-NLS-1$
	
	/** The label of the caption field */
	public static final String LABEL_CAPTION = LaTeXDrawLang.getDialogFrameString("DrawPropertiesFrame.1"); //$NON-NLS-1$
	
	/** The name of the pixelsPerUnitField JFormattedTextField */
	public final static String NAME_PIXELS_FIELD = "pixelsPerUnitField";//$NON-NLS-1$
	
	/** The name of the button ok */
	public static final String NAME_BUTTON_OK = "BUTTON_OK";//$NON-NLS-1$
	
	/** The name of the button cancel */
	public static final String NAME_BUTTON_CANCEL = "BUTTON_CANCEL";//$NON-NLS-1$
	
	/** The title of the frame */
	public static final String TITLE = LaTeXDrawLang.getDialogFrameString("DrawPropertiesFrame.2"); //$NON-NLS-1$
	
	public static final String LABEL_UNIT_PER = LaTeXDrawLang.getDialogFrameString("DrawPropertiesFrame.3"); //$NON-NLS-1$
	
	public static final String ACTION_CMD_UNITCHOICE = "unitChoice"; //$NON-NLS-1$
	
	public static final String LABEL_ONLYFLOATCB = LaTeXDrawLang.getDialogFrameString("DrawPropertiesFrame.5"); //$NON-NLS-1$
	
	public static final String LABEL_TOP = LaTeXDrawLang.getDialogFrameString("DrawPropertiesFrame.6"); //$NON-NLS-1$
	
	public static final String LABEL_BOTTOM = LaTeXDrawLang.getDialogFrameString("DrawPropertiesFrame.7"); //$NON-NLS-1$
	
	public static final String LABEL_HERE = LaTeXDrawLang.getDialogFrameString("DrawPropertiesFrame.8"); //$NON-NLS-1$
	
	public static final String LABEL_HERE_HERE = LaTeXDrawLang.getDialogFrameString("DrawPropertiesFrame.9"); //$NON-NLS-1$
	
	public static final String LABEL_DEFAULT_POSITION = AbstractParametersFrame.LABEL_DEFAULT;
	
	public static final String LABEL_CENTER = LaTeXDrawLang.getDialogFrameString("DrawPropertiesFrame.10"); //$NON-NLS-1$
	
	public static final String TITLE_BORDERS_PLACEMENT_VERT = LaTeXDrawLang.getDialogFrameString("DrawPropertiesFrame.11"); //$NON-NLS-1$
	
	public static final String TITLE_BORDERS_PLACEMENT_HORI = LaTeXDrawLang.getDialogFrameString("DrawPropertiesFrame.12"); //$NON-NLS-1$
	
	
	
	/**
	 * The constructor by default
	 */
	public DrawPropertiesFrame(LaTeXDrawFrame frame)
	{
  		Dimension dim = Toolkit.getDefaultToolkit().getScreenSize();
		parentFrame = frame;
		
  		setIconImage(LaTeXDrawResources.drawingPropIcon.getImage());
  		
  		formerUnitChoice = XScale.DEFAULT_UNIT;
	  	setTitle(TITLE);
  	
	  	JTabbedPane tabbedPane = new JTabbedPane();
  		JButton buttonOk  = new JButton(AbstractParametersFrame.LABEL_OK), 
		buttonCancel  = new JButton(AbstractParametersFrame.LABEL_CANCEL);	
		
  		buttonOk.setActionCommand(NAME_BUTTON_OK);
  		buttonCancel.setActionCommand(NAME_BUTTON_CANCEL);
  		buttonOk.addActionListener(this);
		buttonCancel.addActionListener(this);
		
  		JPanel pButton  = new JPanel();
		pButton.add(buttonOk);
		pButton.add(buttonCancel);
		pButton.setPreferredSize(new Dimension(500, 40));
		pButton.setMaximumSize(new Dimension(500, 40));
		
		tabbedPane.addTab(LaTeXDrawLang.getDialogFrameString("PreferencesFrame.general"), createPanelGenParameters());
		tabbedPane.addTab("Packages", createPanelPackages());
     	BoxLayout bl = new BoxLayout(getContentPane(), BoxLayout.Y_AXIS);
     	getContentPane().setLayout(bl);
     	getContentPane().add(tabbedPane);
     	getContentPane().add(pButton);
  		pack();
  		setLocation(dim.width/2-getWidth()/2, dim.height/2-getHeight()/2);
		setVisible(false);
	}

	
	
	private JPanel createPanelPackages() {
		JPanel pkgP 	= new JPanel();
		JSplitPane pane = new JSplitPane();
		packagesDrawing = new JTextArea();
		packagesList 	= new JTextArea();
		JScrollPane sp1 = new JScrollPane(packagesDrawing);
		JScrollPane sp2 = new JScrollPane(packagesList);
		JPanel p1 		= new JPanel();
		JPanel p2 		= new JPanel();

		packagesList.setEditable(false);
		p1.setBorder(new CompoundBorder(new TitledBorder(null, "Document packages", 
  				TitledBorder.LEFT, TitledBorder.TOP), new EmptyBorder(5,5,5,5)));
		p2.setBorder(new CompoundBorder(new TitledBorder(null, "Packages from preferences", 
				TitledBorder.LEFT, TitledBorder.TOP), new EmptyBorder(5,5,5,5)));
		
		p1.setLayout(new BoxLayout(p1, BoxLayout.X_AXIS));
		p2.setLayout(new BoxLayout(p2, BoxLayout.X_AXIS));
		pkgP.setLayout(new BoxLayout(pkgP, BoxLayout.X_AXIS));
		
		packagesList.setToolTipText("<html>These packages come from your preferences file.<br>They are not used for the drawing.</html>");
		packagesDrawing.setToolTipText("<html>Include into this list the packages you need to compile the drawing, e.g. :<br>\\usepackage[frenchb]{babel}<br>\\usepackage[utf8]{inputenc}<br>You can also copy/paste the packages you already saved in your preferences displayed on the right.</html>");
		
		p1.add(sp1);
		p2.add(sp2);
		pane.setLeftComponent(p1);
		pane.setRightComponent(p2);
		pane.setDividerLocation(230);
		pkgP.add(pane);
		
		return pkgP;
	}
	
	
	protected void setPackagesLists() {
		packagesList.setText(parentFrame.getLatexIncludes());
		packagesDrawing.setText(parentFrame.getCodePanel().getDocumentPackages());
	}
	
	
	
	private JPanel createPanelGenParameters() {
		JPanel pGeneral = new JPanel();
  		pGeneral.setLayout(new BoxLayout(pGeneral, BoxLayout.Y_AXIS));
  		
     	JPanel placementPanel = new JPanel();
     	placementPanel.setLayout(new GridLayout(6, 1));
     	
     	ButtonGroup group = new ButtonGroup();
     	onlyWithFloatsCB = new JRadioButton(LABEL_ONLYFLOATCB);
     	onlyWithFloatsCB.setName(CodePanel.TOKEN_VERT_FLOATS_PAGE);
     	placementPanel.add(onlyWithFloatsCB);
     	group.add(onlyWithFloatsCB);
     	
     	hereCB = new JRadioButton(LABEL_HERE);
     	hereCB.setName(CodePanel.TOKEN_VERT_HERE);
     	placementPanel.add(hereCB);
     	group.add(hereCB);
     	
     	HERECB = new JRadioButton(LABEL_HERE_HERE);
     	placementPanel.add(HERECB);
     	group.add(HERECB);
     	
     	topCB = new JRadioButton(LABEL_TOP);
     	placementPanel.add(topCB);
     	group.add(topCB);
     	
     	bottomCB = new JRadioButton(LABEL_BOTTOM);
     	placementPanel.add(bottomCB);
     	group.add(bottomCB);
     	
     	defaultPositionCB = new JRadioButton(LABEL_DEFAULT_POSITION);
     	placementPanel.add(defaultPositionCB);
     	group.add(defaultPositionCB);
     	
     	placementPanel.setBorder(new CompoundBorder(new TitledBorder(null, TITLE_BORDERS_PLACEMENT_VERT, 
				  				TitledBorder.LEFT, TitledBorder.TOP), new EmptyBorder(5,5,5,5)));
     	pGeneral.add(placementPanel);
     	
     	JPanel placementHoriPanel = new JPanel();
     	placementHoriPanel.setLayout(new GridLayout(2, 1));
     	group = new ButtonGroup();
     	
     	centerHoriCB = new JRadioButton(LABEL_CENTER);
     	placementHoriPanel.add(centerHoriCB);
     	group.add(centerHoriCB);
     	
     	defaultPositionHoriCB = new JRadioButton(LABEL_DEFAULT_POSITION);
     	placementHoriPanel.add(defaultPositionHoriCB);
     	group.add(defaultPositionHoriCB);
     	
     	placementHoriPanel.setBorder(new CompoundBorder(new TitledBorder(null, TITLE_BORDERS_PLACEMENT_HORI, 
				  				TitledBorder.LEFT, TitledBorder.TOP), new EmptyBorder(5,5,5,5)));
     	pGeneral.add(placementHoriPanel);
     	
     	JPanel pPpc = new JPanel(new FlowLayout(FlowLayout.LEFT));
    	SpinnerModel model = new SpinnerNumberModel(parentFrame.getDrawPanel().getPixelsPerCm(),1,100000,1);
     	pixelsPerUnitField = new JSpinner(model);
     	pixelsPerUnitField.setEditor(new JSpinner.NumberEditor(pixelsPerUnitField, "0"));//$NON-NLS-1$
      	pixelsPerUnitField.setToolTipText(LaTeXDrawLang.getDialogFrameString("DrawPropertiesFrame.13")); //$NON-NLS-1$
      	pixelsPerUnitField.setName(NAME_PIXELS_FIELD);
     	pixelsPerUnitField.setMaximumSize(new Dimension(50,30));
     	pPpc.add(pixelsPerUnitField);
     	pPpc.add(new JLabel(LABEL_UNIT_PER));
     	
     	unitChoice = new LaTeXDrawComboBox();
     	unitChoice.addItem(XScale.LABEL_CM);
     	unitChoice.addItem(XScale.LABEL_INCH);
     	unitChoice.setSelectedItem(XScale.DEFAULT_UNIT);
     	unitChoice.setPreferredSize(new Dimension(65,30));
     	unitChoice.setMaximumSize(new Dimension(65,30));
     	unitChoice.setMinimumSize(new Dimension(65,30));
     	unitChoice.setActionCommand(ACTION_CMD_UNITCHOICE);
     	unitChoice.addActionListener(this);
     	pPpc.add(unitChoice);
     	
     	JPanel pCaption = new JPanel(new FlowLayout(FlowLayout.LEFT));
     	pCaption.add(new JLabel(LABEL_CAPTION));
     	captionField = new JTextField();
     	captionField.setMinimumSize(new Dimension(200, 25));
     	captionField.setPreferredSize(new Dimension(200, 25));
     	pCaption.add(captionField);
     	
     	JPanel pLabel = new JPanel(new FlowLayout(FlowLayout.LEFT));
     	pLabel.add(new JLabel(LABEL_LABEL));
     	labelField = new JTextField();
     	labelField.setMinimumSize(new Dimension(200, 25));
     	labelField.setPreferredSize(new Dimension(200, 25));
     	pLabel.add(labelField);
     	
     	pGeneral.add(pPpc);
     	pGeneral.add(pCaption);
     	pGeneral.add(pLabel);
     	
     	return pGeneral;
	}
	
	
	public void actionPerformed(ActionEvent e)
	{
		Object o = e.getSource();
		
		try
		{
			if(o instanceof JButton)
			{
				String label = ((JButton)o).getActionCommand();
				
				if(label == NAME_BUTTON_CANCEL)
				{
					super.setVisible(false);
					return;
				}

				if(label == NAME_BUTTON_OK)
				{
					saveProperties();
					parentFrame.getDrawPanel().repaint();
					setVisible(false);
					return;
				}
			}
			
			if(o instanceof LaTeXDrawComboBox)
			{
				if(((LaTeXDrawComboBox)o).getActionCommand().equals(ACTION_CMD_UNITCHOICE))
				{
					if(!formerUnitChoice.equals(unitChoice.getSelectedItem().toString()))
					{
						float pixf = Float.valueOf(pixelsPerUnitField.getValue().toString());
						
						if(unitChoice.getSelectedItem().toString().equals(XScale.LABEL_CM))
							pixelsPerUnitField.setValue((int)(pixf*PSTricksConstants.INCH_VAL_CM));
						else
							pixelsPerUnitField.setValue((int)(pixf/PSTricksConstants.INCH_VAL_CM+1));
						
						formerUnitChoice = unitChoice.getSelectedItem().toString();
					}
					
					return ;
				}
			}
			
		}catch(Exception ex) 
		{ 
			ex.printStackTrace(); 
			ExceptionFrameDialog.showExceptionDialog(ex);
		}
	}
	
	
	
	/**
	 * Initialise the values of the item of the frame
	 */
	public void setProperties()
	{
		try
		{
			DrawContainer draw = parentFrame.getDrawPanel().getDraw();
		
			setPackagesLists();
			unitChoice.setSelectedItem(draw.getUnitForPixelsPerCm());
			
			int pix = draw.getPixelsPerCm();
			
			if(unitChoice.getSelectedItem().toString().equals(XScale.LABEL_CM))
				pixelsPerUnitField.setValue(pix);
			else
				pixelsPerUnitField.setValue((int)(pix/PSTricksConstants.INCH_VAL_CM+1));
			
			CodePanel codePanel = parentFrame.getCodePanel();
			String currentVertPosition = codePanel.getPositionVertToken();
			String currentHoriPosition = codePanel.getPositionHoriToken();
			
			if(currentVertPosition.equals(CodePanel.TOKEN_VERT_FLOATS_PAGE))
				onlyWithFloatsCB.setSelected(true);
			else if(currentVertPosition.equals(CodePanel.TOKEN_VERT_BOTTOM))
				bottomCB.setSelected(true);
			else if(currentVertPosition.equals(CodePanel.TOKEN_VERT_HERE))
				hereCB.setSelected(true);
			else if(currentVertPosition.equals(CodePanel.TOKEN_VERT_HERE_HERE))
				HERECB.setSelected(true);
			else if(currentVertPosition.equals(CodePanel.TOKEN_VERT_TOP))
				topCB.setSelected(true);
			else defaultPositionCB.setSelected(true);
			
			if(currentHoriPosition.equals(CodePanel.TOKEN_HORI_CENTER_START))
				 centerHoriCB.setSelected(true);
			else defaultPositionHoriCB.setSelected(true);
				
			captionField.setText(codePanel.getCaption());
			labelField.setText(codePanel.getLabel());
			
		}catch(Exception ex)
		{
			ex.printStackTrace(); 
			ExceptionFrameDialog.showExceptionDialog(ex);
		}
	}
	
	
	
	
	@Override
	public void setVisible(boolean visible)
	{
		if(visible)
			setProperties();

		super.setVisible(visible);
	}
	
	
	
	
	/**
	 * Saves the properties
	 */
	public void saveProperties()
	{
		try
		{
			DrawContainer draw = parentFrame.getDrawPanel().getDraw();	
			CodePanel codePanel = parentFrame.getCodePanel();
			float pixf = Float.valueOf(pixelsPerUnitField.getValue().toString());
			int pix = (int)pixf;
			
			if(((String)unitChoice.getSelectedItem()).equals(XScale.LABEL_INCH))
				pix*=PSTricksConstants.INCH_VAL_CM;
	
			parentFrame.setUnitForPixelsPerCm(unitChoice.getSelectedItem().toString());			
			draw.setPixelsPerCm(pix);
			
			codePanel.setLabel(labelField.getText());
			codePanel.setCaption(captionField.getText());
			
			
			if(onlyWithFloatsCB.isSelected())
				codePanel.setPositionVertToken(CodePanel.TOKEN_VERT_FLOATS_PAGE);
			else if(HERECB.isSelected())
				codePanel.setPositionVertToken(CodePanel.TOKEN_VERT_HERE_HERE);
			else if(hereCB.isSelected())
				codePanel.setPositionVertToken(CodePanel.TOKEN_VERT_HERE);
			else if(topCB.isSelected())
				codePanel.setPositionVertToken(CodePanel.TOKEN_VERT_TOP);
			else if(bottomCB.isSelected())
				codePanel.setPositionVertToken(CodePanel.TOKEN_VERT_BOTTOM);
			else if(defaultPositionCB.isSelected())
				codePanel.setPositionVertToken(CodePanel.TOKEN_HORI_VERT_DEFAULT);
			
			if(centerHoriCB.isSelected())
				codePanel.setPositionHoriToken(CodePanel.TOKEN_HORI_CENTER_START);
			else codePanel.setPositionHoriToken(CodePanel.TOKEN_HORI_VERT_DEFAULT);
			
			codePanel.setDocumentPackages(packagesDrawing.getText());
			parentFrame.getDrawPanel().setIsModified(true);
			parentFrame.getDrawPanel().updateCode();
			
		}catch(Exception ex)
		{
			ex.printStackTrace(); 
			ExceptionFrameDialog.showExceptionDialog(ex);
		}
	}
}
