use ui;
use core:geometry;
use graphics;
use lang:bs:macro;

/**
 * The main presentation window.
 */
class Presenter extends Frame {
	// Painter.
	private PresentPainter render;

	// Key mappings.
	private Map<Key, fn()->void> keymap;

	// Any digits entered?
	private Bool anyDigits;

	// Number being entered.
	private Nat number;

	// Source for the presentation. Used to reload the presentation.
	lang:Function? source;

	// Create.
	init(Presentation p) {
		init(p.title, Size(1024, 768)) {
			render(p);
		}

		initKeymap();

		cursorVisible = false;
		painter = render;

		create();
	}

	// Handle keypresses.
	Bool onKey(Bool pressed, Key key, Modifiers modifiers) {
		if (!pressed)
			return false;

		if (modifiers == Modifiers:none) {
			if (keymap.has(key)) {
				keymap.get(key).call();
				return true;
			}
		}

		false;
	}

	// Handle mouse clicks.
	Bool onClick(Bool pressed, Point mousePos, MouseButton button) : override {
		render.onClick(pos.size, mousePos, pressed, button);
	}

	// Double clicks.
	Bool onDblClick(Point mousePos, MouseButton button) : override {
		render.onDblClick(pos.size, mousePos, button);
	}

	// Mouse movement.
	Bool onMouseMove(Point mousePos) : override {
		render.onMouseMove(pos.size, mousePos);
	}

	// Mouse leave.
	void onMouseLeave() {
		render.onMouseLeave();
	}

	// Go to a specific slide.
	void goTo(Nat slide) {
		render.goTo(slide);
		repaint();
	}

	// Cleanup the presentation on close.
	void close() {
		render.cleanup();
		super:close();
	}

private:
	// Initialize the keymap.
	void initKeymap() {
		putKeys([Key:esc, Key:q], &this.close());
		putKeys([Key:right, Key:down], &this.next());
		putKeys([Key:left, Key:up], &this.prev());
		keymap.put(Key:return, &this.enter());
		keymap.put(Key:f, &this.toggleFullscreen());
		keymap.put(Key:F5, &this.reload());
		keymap.put(Key:home, &this.first());
		keymap.put(Key:end, &this.last());
		keymap.put(Key:F8, &debug:threadSummary());

		var me = this;
		keymap.put(Key:num0, () => me.numKey(0));
		keymap.put(Key:num1, () => me.numKey(1));
		keymap.put(Key:num2, () => me.numKey(2));
		keymap.put(Key:num3, () => me.numKey(3));
		keymap.put(Key:num4, () => me.numKey(4));
		keymap.put(Key:num5, () => me.numKey(5));
		keymap.put(Key:num6, () => me.numKey(6));
		keymap.put(Key:num7, () => me.numKey(7));
		keymap.put(Key:num8, () => me.numKey(8));
		keymap.put(Key:num9, () => me.numKey(9));
	}

	// Add a series of keys for the same action.
	void putKeys(Key[] keys, fn()->void fn) {
		for (k in keys)
			keymap.put(k, fn);
	}

	// Toggle fullscreen.
	void toggleFullscreen() {
		fullscreen = !fullscreen;
	}

	// A number has been entered.
	void numKey(Nat num) {
		if (!anyDigits)
			number = 0;

		anyDigits = true;
		number = number * 10 + num;
	}

	// Called when "enter" has been pressed. Either advances one slide or moves to the slide previously entered.
	void enter() {
		if (anyDigits) {
			goTo(number);
			anyDigits = false;
		} else {
			next();
		}
	}

	// Advance to the next slide/animation step.
	void next() {
		render.next();
		repaint();
	}

	// Go back to a previous slide/animation step.
	void prev() {
		render.prev();
		repaint();
	}

	// Go to the first slide.
	void first() {
		render.first();
		repaint();
	}

	// Go to the last slide.
	void last() {
		render.last();
		repaint();
	}

	// Reload the presentation.
	void reload() {
		unless (source)
			return;

		doReload(source);
		if (fn = lang:pointer(source) as Fn<Presentation>) {
			render.presentation = fn.call();
		}
	}
}

// Helper for doing reload.
private void doReload(lang:NameLookup entity) on Compiler {
	// Find containing package.
	if (entity as lang:Package) {
		lang:bs:macro:reloadPkg(entity);
	} else if (parent = entity.parent) {
		// Try to find a parent package...
		doReload(parent);
	}
}


/**
 * Rendering of presentations.
 */
class PresentPainter extends Painter {
	// Current presentation.
	private Presentation p;

	// Currently playing animation?
	private Bool inAnimation;

	// Start of the currently playing animation.
	private Moment aniStart;

	// Length of the currently playing animation.
	private Duration aniLength;

	// Previous state, to keep slide animations consistent.
	private Presentation:Cursor? lastCursor;

	// Our cursor.
	private Presentation:Cursor cursor;

	// Type of pointer to show.
	enum PointerType {
		none,
		highlight,
		arrow,
	}

	// Pointer to show.
	private PointerType pointer;

	// Position of the pointer.
	private Point pointerPos;

	// Brush for the highlight pointer.
	private Brush highlightBrush;

	// Brush for the arrow pointer.
	private Brush arrowBrush;

	// Geometry for the arrow.
	private Path arrowPath;

	// Size of the highlight pointer.
	private Float highlightSize;

	// Mouse pressed?
	private Bool mouseDown;

	// Create.
	init(Presentation p) {
		init() {
			p = p;
			cursor = p.begin;
			highlightBrush = SolidBrush(Color(1, 0, 0, 0.7));
			highlightSize = 10;
			arrowBrush = SolidBrush(Color(0, 0, 0, 0.7));
		}

		arrowPath.start(Point(0, 0));
		arrowPath.line(Point(12, 13));
		arrowPath.line(Point(5, 13));
		arrowPath.line(Point(0, 18));
		arrowPath.close();

		bgColor = black;

		// Notify the presentation that it can repaint things!
		p.setupRepaint(&this.repaint());

		animate();
	}

	// Draw things!
	Bool render(Size size, Graphics g) {
		// How is the animation going?
		Float aniPos = 1.0;
		if (inAnimation) {
			aniPos = (Moment() - aniStart) / aniLength;
			if (aniPos >= 1.0) {
				inAnimation = false;
				aniPos = 1.0;
			}
		}

		// Compute where the contents are supposed to be.
		Size content = p.size;
		Float scale = min(size.w / content.w, size.h / content.h);
		Point move = (size - content*scale) / 2;
		g.transform = scale(scale) * translate(move);
		g.push(Rect(content));

		p.draw(g, cursor, lastCursor, aniPos);

		g.pop();

		if (pointer == PointerType:highlight) {
			g.fillOval(Rect(pointerPos - Size(highlightSize), Size(highlightSize*2)), highlightBrush);
		} else if (pointer == PointerType:arrow) {
			g.push();
			g.transform = translate(pointerPos);
			g.fill(arrowPath, arrowBrush);
			g.pop();
		}

		inAnimation;
	}

	// Translate from screen coordinates into presentation coordinates.
	private Point fromScreen(Size size, Point pt) {
		Size content = p.size;
		Float scale = min(size.w / content.w, size.h / content.h);
		Point move = (size - content*scale) / 2;
		return (pt - move) / scale;
	}

	// Handle mouse clicks.
	Bool onClick(Size windowSize, Point pos, Bool pressed, MouseButton button) {
		pos = fromScreen(windowSize, pos);

		Bool doRepaint = false;
		if (button == MouseButton:left) {
			doRepaint |= mouseDown != pressed;
			mouseDown = pressed;
			if (mouseDown) {
				pointer = PointerType:highlight;
				pointerPos = pos;
			} else {
				pointer = PointerType:none;
			}
		}

		if (element = p.elementAt(cursor, pos)) {
			if (interactive = element.interactive()) {
				doRepaint = interactive.mouseClicked(pos, pressed, button);
				pointer = PointerType:arrow;
			}
		}

		if (doRepaint)
			repaint();

		true;
	}

	// Handle double clicks.
	Bool onDblClick(Size windowSize, Point pos, MouseButton button) {
		pos = fromScreen(windowSize, pos);

		if (element = p.elementAt(cursor, pos)) {
			if (interactive = element.interactive()) {
				if (interactive.mouseDblClicked(pos, button))
					repaint();
			}
		}

		true;
	}

	// Mouse movement.
	Bool onMouseMove(Size windowSize, Point pos) {
		pos = fromScreen(windowSize, pos);

		Bool needRepaint = pointer != PointerType:none;
		if (mouseDown) {
			pointer = PointerType:highlight;
			pointerPos = pos;
			needRepaint = true;
		} else {
			pointer = PointerType:none;
		}

		if (element = p.elementAt(cursor, pos)) {
			if (interactive = element.interactive()) {
				needRepaint |= interactive.mouseMoved(pos);
				pointer = PointerType:arrow;
				pointerPos = pos;
				needRepaint = true;
			}
		}

		if (needRepaint)
			repaint();
		true;
	}

	// Mouse leave.
	void onMouseLeave() {
		Bool needRepaint = pointer != PointerType:none;
		mouseDown = false;
		pointer = PointerType:none;
		if (needRepaint)
			repaint();
		true;
	}

	// Advance the animation.
	void next() {
		setCursor(p.next(cursor));
	}

	// Go back.
	void prev() {
		setCursor(p.prev(cursor));
	}

	// Go to specific slide.
	void goTo(Nat slide) {
		setCursor(p.at(slide));
	}

	// Go to the first slide.
	void first() {
		setCursor(p.at(0));
	}

	// Go to the last (interesting) slide.
	void last() {
		setCursor(p.prev(p.end));
	}

	// Set the presentation.
	assign presentation(Presentation pr) {
		p.cleanup();
		p = pr;
		p.setupRepaint(&this.repaint());
		repaint();
	}

	// Cleanup the current presentation.
	void cleanup() {
		p.cleanup();
	}

	// Set the cursor to a new location.
	private void setCursor(Presentation:Cursor c) {
		lastCursor = cursor;
		cursor = c;
		animate();
	}

	// Set us up for a new animation specified by the current cursor.
	private void animate() {
		var d = p.duration(cursor);

		if (d > 0 ms) {
			inAnimation = true;
			aniStart = Moment();
			aniLength = d;
		} else {
			inAnimation = false;
		}
	}
}

// Make sure the entire presentation library is compiled.
void ensureCompiled() {
	print("Compiling...");
	Moment start;
	compile(named{presentation});
	compile(named{layout});
	Moment end;
	print("Done in ${end - start}");
}

// Helper for showing a presentation.
void show(Presentation p) on Ui {
	ensureCompiled();
	Presenter w(p);
	w.waitForClose();
}

// Helper for showing a presentation, starts at the specified slide. Good when developing the presentation.
void show(Presentation p, Nat slide) on Ui {
	ensureCompiled();
	Presenter w(p);
	w.goTo(slide);
	w.waitForClose();
}

// Show a presentation in debug mode, allowing live-reloading changes to the presentation.
void showDebug(lang:Function presentation) on Ui {
	if (fn = lang:pointer(presentation) as Fn<Presentation>) {
		Presenter w(fn.call());
		w.source = presentation;
		w.waitForClose();
	} else {
		print("Error: Not a presentation function!");
	}
}

// Show a presentation in debug mode, allowing live-reloading changes to the presentation.
void showDebug(lang:Function presentation, Nat slide) on Ui {
	if (fn = lang:pointer(presentation) as Fn<Presentation>) {
		Presenter w(fn.call());
		w.source = presentation;
		w.goTo(slide);
		w.waitForClose();
	} else {
		print("Error: Not a presentation function!");
	}
}
