/*
 * Copyright 2021-2022 the original author or authors.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      https://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package org.opentest4j.reporting.events.api;

import java.util.function.Consumer;

/**
 * Factory for creating elements of a certain type for a given {@linkplain Context context}.
 *
 * @param <T> the type of element to create
 */
@FunctionalInterface
public interface Factory<T> {

	/**
	 * Create an element using the supplied {@linkplain Context context}.
	 *
	 * @param context the context to use
	 * @return the created element
	 */
	T create(Context context);

	/**
	 * Create and configure an element using the supplied {@linkplain Context context} and {@link Consumer configurer}.
	 *
	 * @param context    the context to use
	 * @param configurer the configurer to use
	 * @return the created element
	 */
	default T createAndConfigure(Context context, Consumer<? super T> configurer) {
		T result = create(context);
		configurer.accept(result);
		return result;
	}
}
