/*
 * base64.c -- base-64 conversion routines.
 *
 * For license terms, see the file COPYING in this directory.
 *
 * This base 64 encoding is defined in RFC2045 section 6.8,
 * "Base64 Content-Transfer-Encoding", but lines must not be broken in the
 * scheme used here.
 */
#include "config.h"
#include "fetchmail.h"
#include <ctype.h>

static const char base64digits[] =
   "ABCDEFGHIJKLMNOPQRSTUVWXYZabcdefghijklmnopqrstuvwxyz0123456789+/";

#define BAD	-1
static const char base64val[] = {
    BAD,BAD,BAD,BAD, BAD,BAD,BAD,BAD, BAD,BAD,BAD,BAD, BAD,BAD,BAD,BAD,
    BAD,BAD,BAD,BAD, BAD,BAD,BAD,BAD, BAD,BAD,BAD,BAD, BAD,BAD,BAD,BAD,
    BAD,BAD,BAD,BAD, BAD,BAD,BAD,BAD, BAD,BAD,BAD, 62, BAD,BAD,BAD, 63,
     52, 53, 54, 55,  56, 57, 58, 59,  60, 61,BAD,BAD, BAD,BAD,BAD,BAD,
    BAD,  0,  1,  2,   3,  4,  5,  6,   7,  8,  9, 10,  11, 12, 13, 14,
     15, 16, 17, 18,  19, 20, 21, 22,  23, 24, 25,BAD, BAD,BAD,BAD,BAD,
    BAD, 26, 27, 28,  29, 30, 31, 32,  33, 34, 35, 36,  37, 38, 39, 40,
     41, 42, 43, 44,  45, 46, 47, 48,  49, 50, 51,BAD, BAD,BAD,BAD,BAD
};
#define DECODE64(c)  (isascii((unsigned char)(c)) ? base64val[c] : BAD)

int to64frombits(char *out, const void *in_, int inlen, size_t outlen)
/* raw bytes in quasi-big-endian order to base 64 string (NUL-terminated) */
{
    int rc = 0;
    const unsigned char *in = (const unsigned char *)in_;

    for (; inlen >= 3; inlen -= 3)
    {
	if (outlen < 5) { rc = -1; goto fail; } /* buffer too small */
	*out++ = base64digits[in[0] >> 2];
	*out++ = base64digits[((in[0] << 4) & 0x30) | (in[1] >> 4)];
	*out++ = base64digits[((in[1] << 2) & 0x3c) | (in[2] >> 6)];
	*out++ = base64digits[in[2] & 0x3f];
	in += 3;
	outlen -= 4;
    }
    if (inlen > 0)
    {
	unsigned char fragment;
    
	if (outlen < 5) { rc = -1; goto fail; } /* buffer too small */
	*out++ = base64digits[in[0] >> 2];
	fragment = (in[0] << 4) & 0x30;
	if (inlen > 1)
	    fragment |= in[1] >> 4;
	*out++ = base64digits[fragment];
	*out++ = (inlen < 2) ? '=' : base64digits[(in[1] << 2) & 0x3c];
	*out++ = '=';
    }
fail:
    *out = '\0';
    return rc;
}

int from64tobits(void *out_, const char *in, int maxlen)
/* base 64 to raw bytes in quasi-big-endian order, returning count of bytes */
/* maxlen limits output buffer size, set to zero to ignore */
{
    int len = 0;
    unsigned char digit1, digit2, digit3, digit4;
    unsigned char *out = (unsigned char *)out_;

    if (in[0] == '+' && in[1] == ' ')
	in += 2;
    if (*in == '\r')
	return(0);

    do {
	digit1 = in[0];
	if (DECODE64(digit1) == BAD)
	    return(-1);
	digit2 = in[1];
	if (DECODE64(digit2) == BAD)
	    return(-1);
	digit3 = in[2];
	if (digit3 != '=' && DECODE64(digit3) == BAD)
	    return(-1); 
	digit4 = in[3];
	if (digit4 != '=' && DECODE64(digit4) == BAD)
	    return(-1);
	in += 4;
	++len;
	if (maxlen && len > maxlen)
	    return(-1);
	*out++ = (DECODE64(digit1) << 2) | (DECODE64(digit2) >> 4);
	if (digit3 != '=')
	{
	    ++len;
	    if (maxlen && len > maxlen)
	        return(-1);
	    *out++ = ((DECODE64(digit2) << 4) & 0xf0) | (DECODE64(digit3) >> 2);
	    if (digit4 != '=')
	    {
	        ++len;
		if (maxlen && len > maxlen)
		    return(-1);
		*out++ = ((DECODE64(digit3) << 6) & 0xc0) | DECODE64(digit4);
	    }
	}
    } while 
	(*in && *in != '\r' && digit4 != '=');

    return len;
}

/* base64.c ends here */
